﻿#include "type.h"          // 基础类型（优先包含，保证类型统一）
#include "net_interface.h" // 抽象层接口（依赖基础类型，后于type.h）
#include "nms_snmp_api.h"  // 错误码定义（显式包含，避免隐式依赖）
#include "agent_log.h"     // 日志工具
#include <string.h>

// 假设的专用网络SDK头文件（预留，实际实现时替换）
// #include "es_sdk.h"

// 预留：ES接口句柄（void* 屏蔽SDK细节，无硬件依赖）
static void* g_es_handle = NULL;

/**
 * 参数合法性校验（与ETH接口保持一致，统一校验标准）
 */
static bool8 net_validate_params(const char* ip, uint16 port, const uint8* data, uint32 len) {
    // 补充IP格式简单校验（避免空指针或无效IP字符串）
    if (ip == NULL || strlen(ip) == 0 || strlen(ip) >= API_IP_LEN) {
        return BOOL8_FALSE;
    }
    // 端口范围（1-65535）、数据长度（1-MAX_MSG_LEN）校验
    return (port > 0 && port <= 65535 &&
            data != NULL && len > 0 && len <= MAX_MSG_LEN);
}

/**
 * ES销毁函数（预留实现，确保资源释放幂等性）
 */
static void es_destroy(void) {
    if (g_es_handle == NULL) {
        return; // 避免重复销毁
    }

    // 预留：实际实现时补充SDK关闭逻辑（需判断SDK销毁返回值）
    // if (es_sdk_close(g_es_handle) != ES_SUCCESS) {
    //     agent_log_error(__func__, "ES SDK关闭失败");
    // }

    g_es_handle = NULL;
    agent_log_info(__func__, "ES接口资源已释放（预留）");
}

/**
 * ES发送实现（预留，与NetInterface.net_send原型严格对齐）
 */
static ApiSnmpError es_send(const char* ip, uint16 port, const uint8* data, uint32 len) {
    // 参数校验（优先返回无效参数，错误定位更清晰）
    if (!net_validate_params(ip, port, data, len)) {
        agent_log_error(__func__, "ES发送参数无效（ip：%s，port：%d，len：%d）", ip, port, len);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 检查接口是否已初始化
    if (g_es_handle == NULL) {
        agent_log_error(__func__, "ES接口未初始化");
        return API_SNMP_ERR_NETWORK;
    }

    // 预留：实际实现时补充SDK发送逻辑
    // ES_Result result = es_sdk_send(g_es_handle, ip, port, data, len);
    // if (result != ES_SUCCESS) {
    //     agent_log_error(__func__, "ES SDK发送失败（错误码：%d）", result);
    //     return API_SNMP_ERR_NETWORK;
    // }

    agent_log_warn(__func__, "ES发送接口未实现（预留）：%d字节到%s:%d", len, ip, port);
    return API_SNMP_ERR_OP_NOT_SUPPORTED;
}

/**
 * ES接收实现（预留，与NetInterface.net_recv原型严格对齐）
 */
static ApiSnmpError es_recv(uint8* buf, uint32 buf_len, uint32* recv_len,
                           char* src_ip, uint16* src_port) {
    // 参数校验（避免空指针、缓冲区长度为0）
    if (buf == NULL || buf_len == 0 || recv_len == NULL ||
        src_ip == NULL || src_port == NULL) {
        agent_log_error(__func__, "ES接收参数无效");
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 检查接口是否已初始化
    if (g_es_handle == NULL) {
        agent_log_error(__func__, "ES接口未初始化");
        return API_SNMP_ERR_NETWORK;
    }

    // 初始化输出参数（避免上层使用未初始化的值）
    *recv_len = 0;
    memset(src_ip, 0, API_IP_LEN);
    *src_port = 0;

    // 预留：实际实现时补充SDK接收逻辑（内部固定超时1秒，与ETH保持一致）
    // ES_Result result = es_sdk_recv(g_es_handle, buf, buf_len, 1000, src_ip, src_port, recv_len);
    // if (result == ES_TIMEOUT) {
    //     agent_log_debug(__func__, "ES接收超时");
    //     return API_SNMP_ERR_TIMEOUT;
    // } else if (result != ES_SUCCESS) {
    //     agent_log_error(__func__, "ES SDK接收失败（错误码：%d）", result);
    //     return API_SNMP_ERR_NETWORK;
    // }

    agent_log_warn(__func__, "ES接收接口未实现（预留）");
    return API_SNMP_ERR_OP_NOT_SUPPORTED;
}

/**
 * 初始化ES接口（预留注册，确保幂等性）
 */
ApiSnmpError net_es_init(void) {
    // 避免重复初始化（幂等设计，支持多次调用）
    if (g_es_handle != NULL) {
        agent_log_info(__func__, "ES接口已初始化（预留），无需重复操作");
        return API_SNMP_ERR_SUCCESS;
    }

    // 预留：实际实现时补充SDK初始化逻辑（含参数校验）
    // ES_InitParam param = {0};
    // param.timeout = 5000; // 5秒超时
    // param.max_buf_len = MAX_MSG_LEN;
    // g_es_handle = es_sdk_open(&param);
    // if (g_es_handle == NULL) {
    //     agent_log_error(__func__, "ES SDK初始化失败");
    //     return API_SNMP_ERR_NETWORK;
    // }

    // 标记为已初始化（非NULL值即可，避免与NULL混淆）
    g_es_handle = (void*)1;

    // 注册ES接口到抽象层（填充完整结构体，与抽象层定义一致）
    NetInterface es_iface = {
        .type = NET_IFACE_ES,        // 接口类型（与枚举严格匹配）
        .net_send = es_send,         // 绑定发送函数
        .net_recv = es_recv,         // 绑定接收函数
        .destroy = es_destroy,       // 绑定销毁函数（确保资源释放）
        .local_port = 0              // ES无需绑定固定端口，设为0
    };

    // 注册到抽象层（返回注册结果，便于上层排查）
    ApiSnmpError err = net_interface_register(&es_iface);
    if (err == API_SNMP_ERR_SUCCESS) {
        agent_log_info(__func__, "ES接口注册成功（预留）");
    } else {
        agent_log_error(__func__, "ES接口注册失败（错误：%d）", err);
        // 注册失败时释放已分配资源（避免内存泄漏）
        es_destroy();
    }

    return err;
}
