﻿#include "type.h"          // 基础类型（优先包含，保证类型统一）
#include "net_interface.h" // 上层抽象接口（依赖赖基础类型）
#include "platform.h"      // 平台抽象接口（Windows 适配）
#include "nms_snmp_api.h"  // 错误码定义（显式包含）
#include "agent_log.h"     // 日志工具
#include <winsock2.h>      // Windows 网络相关头文件
#include <ws2tcpip.h>      // IP 地址转换函数
#include <windows.h>       // Windows 系统 API
#include <string.h>

// 链接 Windows 网络库
#pragma comment(lib, "ws2_32.lib")

// -------------------------- 静态资源（仅当前文件可见）--------------------------
static PlatformSocket g_eth_socket = PLATFORM_SOCKET_INVALID;  // ETH 套接字句柄
static HANDLE g_interrupt_thread = NULL;                       // 接收中断线程句柄
static volatile bool8 g_thread_running = BOOL8_FALSE;          // 线程运行标志
static PlatformMutex g_sock_set_mutex = PLATFORM_MUTEX_INVALID;// 套接字集合锁（线程安全）

// Windows 专用：套接字集合管理（仅 ETH 模块使用）
static fd_set g_eth_socks;
static int32 g_eth_max_sock = -1;

// -------------------------- 内部工具函数（静态隐藏）--------------------------
/**
 * 初始化套接字集合锁（确保线程安全）
 */
static ApiSnmpError eth_init_sock_mutex(void) {
    if (g_sock_set_mutex != PLATFORM_MUTEX_INVALID) {
        return API_SNMP_ERR_SUCCESS;
    }

    g_sock_set_mutex = platform_mutex_create();
    if (g_sock_set_mutex == PLATFORM_MUTEX_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 创建套接字集合锁失败", __func__);
        return API_SNMP_ERR_RESOURCE;
    }
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 检查端口是否已被占用（Windows 平台实现）
 */
static bool8 eth_check_port_in_use(uint16 port) {
    // 创建临时套接字测试端口可用性
    SOCKET test_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (test_sock == INVALID_SOCKET) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 创建测试套接字失败（错误码：%d）", __func__, WSAGetLastError());
        return BOOL8_TRUE; // 视为占用，避免误判
    }

    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port = htons(port);

    // 绑定失败说明端口已占用
    bool8 is_used = BOOL8_TRUE;
    if (bind(test_sock, (struct sockaddr*)&addr, sizeof(addr)) == 0) {
        is_used = BOOL8_FALSE;
    }

    closesocket(test_sock);
    return is_used;
}

// -------------------------- 套接字集合管理接口（供抽象层调用）--------------------------
/**
 * 初始化 ETH 专用套接字集合
 */
void eth_socket_set_init(void) {
    // 初始化线程安全锁
    if (eth_init_sock_mutex() != API_SNMP_ERR_SUCCESS) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 套接字集合初始化失败（锁创建失败）", __func__);
        return;
    }

    platform_mutex_lock(g_sock_set_mutex);
    FD_ZERO(&g_eth_socks);
    g_eth_max_sock = -1;
    platform_mutex_unlock(g_sock_set_mutex);
    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_debug("%s: ETH 套接字集合初始化完成", __func__);
}

/**
 * 添加套接字到集合（线程安全）
 */
void eth_socket_set_add(int32 sock_fd) {
    if (sock_fd == PLATFORM_SOCKET_INVALID || g_sock_set_mutex == PLATFORM_MUTEX_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 添加失败：无效套接字或未初始化锁", __func__);
        return;
    }

    platform_mutex_lock(g_sock_set_mutex);
    FD_SET((SOCKET)sock_fd, &g_eth_socks);
    if (sock_fd > g_eth_max_sock) {
        g_eth_max_sock = sock_fd;
    }
    platform_mutex_unlock(g_sock_set_mutex);
    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_debug("%s: 套接字%d添加到ETH集合", __func__, sock_fd);
}

/**
 * 从集合移除套接字（线程安全）
 */
void eth_socket_set_remove(int32 sock_fd) {
    if (sock_fd == PLATFORM_SOCKET_INVALID || g_sock_set_mutex == PLATFORM_MUTEX_INVALID) {
        return;
    }

    platform_mutex_lock(g_sock_set_mutex);
    FD_CLR((SOCKET)sock_fd, &g_eth_socks);
    // 若移除的是最大套接字，重新查找最大值（简化实现）
    if (sock_fd == g_eth_max_sock) {
        g_eth_max_sock = -1;
        for (int32 i = 0; i <= (int32)FD_SETSIZE; i++) {
            if (FD_ISSET(i, &g_eth_socks) && i > g_eth_max_sock) {
                g_eth_max_sock = i;
            }
        }
    }
    platform_mutex_unlock(g_sock_set_mutex);
    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_debug("%s: 套接字%d从ETH集合移除", __func__, sock_fd);
}

/**
 * 检查套接字是否在集合中（线程安全）
 */
bool8 eth_socket_set_isset(int32 sock_fd) {
    if (sock_fd == PLATFORM_SOCKET_INVALID || g_sock_set_mutex == PLATFORM_MUTEX_INVALID) {
        return BOOL8_FALSE;
    }

    bool8 result = BOOL8_FALSE;
    platform_mutex_lock(g_sock_set_mutex);
    result = (FD_ISSET((SOCKET)sock_fd, &g_eth_socks)) ? BOOL8_TRUE : BOOL8_FALSE;
    platform_mutex_unlock(g_sock_set_mutex);
    return result;
}

// -------------------------- ETH 核心功能实现 --------------------------
/**
 * 绑定本地动态端口（由系统自动分配，1024+）
 */
static ApiSnmpError eth_bind_dynamic_port(PlatformSocket sock) {
    // 绑定0端口表示由系统分配动态端口
    if (!platform_udp_bind(sock, 0)) {
        int err_code = WSAGetLastError();
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 绑定动态端口失败（错误码：%d）", __func__, err_code);
        return API_SNMP_ERR_NETWORK;
    }

    // 获取系统实际分配的端口（用于日志确认）
    struct sockaddr_in addr;
    socklen_t addr_len = sizeof(addr);
    if (getsockname((SOCKET)sock, (struct sockaddr*)&addr, &addr_len) == 0) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_info("%s: 已绑定动态端口：%d", __func__, ntohs(addr.sin_port));
    }

    return API_SNMP_ERR_SUCCESS;
}

/**
 * ETH 发送实现（适配抽象层 net_send 函数指针）
 */
static ApiSnmpError eth_send(const char* ip, uint16 port, const uint8* data, uint32 len) {
    // 参数校验
    if (g_eth_socket == PLATFORM_SOCKET_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 发送失败：ETH 套接字未初始化", __func__);
        return API_SNMP_ERR_NETWORK;
    }
    if (ip == NULL || strlen(ip) == 0 || strlen(ip) >= API_IP_LEN ||
        port == 0 || port > 65535 || data == NULL || len == 0 || len > MAX_MSG_LEN) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 发送失败：无效参数（ip：%s，port：%d，len：%d）", __func__, ip, port, len);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 调用平台发送接口
    int32 ret = platform_udp_sendto(g_eth_socket, ip, port, data, len);
    if (ret <= 0 || (uint32)ret != len) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 发送失败（实际发送：%d字节，预期：%d字节，错误码：%d）",
                       __func__, ret, len, WSAGetLastError());
        return API_SNMP_ERR_NETWORK;
    }

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_debug("%s: ETH 发送成功：%d字节到%s:%d", __func__, len, ip, port);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * ETH 接收实现（适配抽象层 net_recv 函数指针）
 */
static ApiSnmpError eth_recv(uint8* buf, uint32 buf_len, uint32* recv_len,
                            char* src_ip, uint16* src_port) {
    // 参数校验
    if (g_eth_socket == PLATFORM_SOCKET_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 接收失败：ETH 套接字未初始化", __func__);
        return API_SNMP_ERR_NETWORK;
    }
    if (buf == NULL || buf_len == 0 || recv_len == NULL ||
        src_ip == NULL || src_port == NULL) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 接收失败：无效参数", __func__);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 初始化输出参数
    *recv_len = 0;
    memset(src_ip, 0, API_IP_LEN);
    *src_port = 0;

    // 调用平台接收接口（超时1秒）
    int32 ret = platform_udp_recvfrom(g_eth_socket, buf, buf_len, src_ip, src_port, 1000);
    if (ret < 0) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 接收失败（错误码：%d）", __func__, WSAGetLastError());
        return API_SNMP_ERR_NETWORK;
    } else if (ret == 0) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_debug("%s: 接收超时", __func__);
        return API_SNMP_ERR_TIMEOUT;
    }

    *recv_len = (uint32)ret;
    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_debug("%s: ETH 接收成功：%d字节来自%s:%d", __func__, *recv_len, src_ip, *src_port);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 中断线程：持续接收数据并通过抽象层入队
 */
static DWORD WINAPI eth_interrupt_thread_proc(LPVOID param) {
    uint8 recv_buf[MAX_MSG_LEN] = {0};
    struct sockaddr_in src_addr;
    socklen_t src_addr_len = sizeof(src_addr);
    int32 recv_len = 0;
    fd_set read_fds;       // 临时套接字集合（用于select）
    struct timeval timeout;// 超时控制（避免线程阻塞）

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: ETH 接收中断线程启动（监听动态端口）", __func__);

    while (g_thread_running) {
        // 1. 复制当前活跃的套接字集合（加锁保护，避免并发修改）
        platform_mutex_lock(g_sock_set_mutex);
        read_fds = g_eth_socks;
        platform_mutex_unlock(g_sock_set_mutex);

        // 2. 设置超时（100ms，确保线程能响应退出信号）
        timeout.tv_sec = 0;
        timeout.tv_usec = 100000;

        // 3. 等待套接字有数据可读（监听所有动态端口）
        int32 ready = select(g_eth_max_sock + 1, &read_fds, NULL, NULL, &timeout);
        if (ready < 0) {
            // 修复：日志格式统一为 "函数名: 内容"
            agent_log_error("%s: select失败（错误码：%d），线程退出", __func__, WSAGetLastError());
            break;
        } else if (ready == 0) {
            continue; // 超时，继续循环
        }

        // 4. 遍历所有有数据的套接字，处理响应
        platform_mutex_lock(g_sock_set_mutex);
        for (int32 sock_fd = 0; sock_fd <= g_eth_max_sock; sock_fd++) {
            if (FD_ISSET(sock_fd, &read_fds)) {
                // 从动态端口接收响应（与发送端口一致）
                src_addr_len = sizeof(src_addr);
                recv_len = recvfrom(
                    sock_fd, 
                    (char*)recv_buf, 
                    MAX_MSG_LEN, 
                    0, 
                    (struct sockaddr*)&src_addr, 
                    &src_addr_len
                );

                if (recv_len > 0) {
                    // 解析源IP和端口，入队处理
                    char src_ip[API_IP_LEN] = {0};
                    inet_ntop(AF_INET, &src_addr.sin_addr, src_ip, API_IP_LEN);
                    uint16 src_port = ntohs(src_addr.sin_port);

                    NetRecvNode node;
                    memcpy(node.data, recv_buf, recv_len);
                    node.len = (uint32)recv_len;
                    platform_strcpy_s(node.src_ip, API_IP_LEN, src_ip);
                    node.src_port = src_port;

                    ApiSnmpError err = net_interface_enqueue(&node);
                    if (err != API_SNMP_ERR_SUCCESS) {
                        // 修复：日志格式统一为 "函数名: 内容"
                        agent_log_warn("%s: 数据入队失败（错误：%d），丢弃%d字节来自%s:%d",
                                      __func__, err, recv_len, src_ip, src_port);
                    } else {
                        // 修复：日志格式统一为 "函数名: 内容"
                        agent_log_debug("%s: 数据入队成功（%d字节来自%s:%d，端口：%d）",
                                      __func__, recv_len, src_ip, src_port, sock_fd);
                    }

                    memset(recv_buf, 0, recv_len); // 仅清空有效数据区
                }
            }
        }
        platform_mutex_unlock(g_sock_set_mutex);
    }

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: ETH 接收中断线程退出", __func__);
    return 0;
}

/**
 * 销毁 ETH 资源（线程+套接字+锁）
 */
static void eth_destroy(void) {
    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: 开始销毁ETH接口资源", __func__);

    // 1. 先关闭套接字（确保recvfrom立即返回错误）
    if (g_eth_socket != PLATFORM_SOCKET_INVALID) {
        platform_socket_close(g_eth_socket);
        g_eth_socket = PLATFORM_SOCKET_INVALID;
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_info("%s: ETH 套接字已关闭", __func__);
    }

    // 2. 再停止接收线程
    if (g_thread_running) {
        g_thread_running = BOOL8_FALSE;
        if (g_interrupt_thread != NULL) {
            WaitForSingleObject(g_interrupt_thread, 1000); // 等待线程退出
            CloseHandle(g_interrupt_thread);
            g_interrupt_thread = NULL;
            // 修复：日志格式统一为 "函数名: 内容"
            agent_log_info("%s: ETH 接收线程已销毁", __func__);
        }
    }

    // 3. 清理其他资源（锁、集合）
    if (g_sock_set_mutex != PLATFORM_MUTEX_INVALID) {
        platform_mutex_destroy(g_sock_set_mutex);
        g_sock_set_mutex = PLATFORM_MUTEX_INVALID;
    }
    FD_ZERO(&g_eth_socks);
    g_eth_max_sock = -1;

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: ETH 接口资源销毁完成", __func__);
}

// -------------------------- 对外接口（注册到抽象层）--------------------------
/**
 * 初始化 ETH 接口（Windows 平台）并注册到抽象层
 */
ApiSnmpError net_eth_win_init(void) {
    // 避免重复初始化
    if (g_eth_socket != PLATFORM_SOCKET_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_info("%s: ETH 接口已初始化，无需重复操作", __func__);
        return API_SNMP_ERR_SUCCESS;
    }

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: 开始初始化ETH接口（Windows平台，动态端口模式）", __func__);

    // 1. 创建 UDP 套接字
    g_eth_socket = platform_udp_socket_create();
    if (g_eth_socket == PLATFORM_SOCKET_INVALID) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 创建UDP套接字失败（错误码：%d）", __func__, WSAGetLastError());
        return API_SNMP_ERR_NETWORK;
    }

    // 2. 绑定动态端口（替代原161端口绑定）
    ApiSnmpError err = eth_bind_dynamic_port(g_eth_socket);
    if (err != API_SNMP_ERR_SUCCESS) {
        platform_socket_close(g_eth_socket);
        g_eth_socket = PLATFORM_SOCKET_INVALID;
        return err;
    }

    // 3. 添加套接字到集合管理
    eth_socket_set_add((int32)g_eth_socket);

    // 4. 启动接收中断线程
    g_thread_running = BOOL8_TRUE;
    g_interrupt_thread = CreateThread(
        NULL,                   // 默认安全属性
        0,                      // 默认栈大小
        eth_interrupt_thread_proc, // 线程函数
        NULL,                   // 无参数
        0,                      // 立即执行
        NULL                    // 不获取线程ID
    );
    if (g_interrupt_thread == NULL) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: 创建接收线程失败（错误码：%d）", __func__, GetLastError());
        eth_destroy(); // 销毁已创建的资源
        return API_SNMP_ERR_INIT;
    }

    // 注册到抽象层（修正后）
    NetInterface eth_iface = {
        .type = NET_IFACE_ETH,
        .net_send = eth_send,
        .net_recv = eth_recv,
        .destroy = eth_destroy,
        .local_port = 0 // 动态端口，此处填0
    };
    err = net_interface_register(&eth_iface);
    if (err != API_SNMP_ERR_SUCCESS) {
        // 修复：日志格式统一为 "函数名: 内容"
        agent_log_error("%s: ETH 接口注册到抽象层失败（错误：%d）", __func__, err);
        eth_destroy();
        return err;
    }

    // 修复：日志格式统一为 "函数名: 内容"
    agent_log_info("%s: ETH 接口初始化完成（套接字句柄：%d）", __func__, g_eth_socket);
    return API_SNMP_ERR_SUCCESS;
}
