﻿// 头文件包含顺序：基础类型 → 平台接口 → 自身接口 → 工具/标准库
#include "type.h"          // 基础类型（必须最先包含）
#include "platform.h"      // 平台操作接口（仅.c可见，隐藏同步细节）
#include "net_interface.h" // 自身接口声明
#include "nms_snmp_api.h"  // 错误码定义（ApiSnmpError等）
#include "agent_log.h"     // 日志工具
#include <string.h>        // 字符串函数
#include <stdlib.h>

// --------------- 内部隐藏结构体 定义（完全隐藏，不暴露给头文件）
// 接收队列节点（仅内部使用，含链表指针）
typedef struct InternalRecvNode {
    uint8 data[MAX_MSG_LEN];
    uint32 len;
    char src_ip[API_IP_LEN];
    uint16 src_port;
    struct InternalRecvNode* next;
} InternalRecvNode;

// 接收队列（含同步机制，完全隐藏）
typedef struct {
    InternalRecvNode* head;
    InternalRecvNode* tail;
    uint32 count;
    uint32 max_count;
    void* mutex;  // 通用互斥锁句柄（平台无关）
    void* sem;    // 通用信号量句柄（平台无关）
} InternalRecvQueue;
#if 1
// 全局状态变量（静态隐藏）
static NetInterface g_registered_ifaces[NET_IFACE_MAX] = {0};  // 已注册的接口数组
static NetInterfaceType g_current_type = NET_IFACE_ETH;          // 当前激活的接口类型
static NetInterface* g_current_iface = NULL;                     // 当前接口指针
static InternalRecvQueue g_recv_queue;                           // 接收队列实例（隐藏同步细节）
static NetRecvCallback g_recv_callback = NULL;                   // 接收回调函数
static void* g_callback_user_data = NULL;                        // 回调用户数据
static bool8 g_is_inited = BOOL8_FALSE;                          // 初始化标志

/**
 * 初始化接收队列（内部函数，隐藏同步细节）
 */
static void net_interface_init_queue(uint32 max_count) {
    if (max_count == 0) max_count = 100;  // 默认最大100个节点
    g_recv_queue.head = NULL;
    g_recv_queue.tail = NULL;
    g_recv_queue.count = 0;
    g_recv_queue.max_count = max_count;
    // 创建互斥锁（平台接口返回void*）
    g_recv_queue.mutex = platform_mutex_create();
    if (!g_recv_queue.mutex) {
        agent_log_error("[%s] 创建互斥锁失败", __func__);
        return;
    }

    // 创建信号量（平台接口返回void*）
    g_recv_queue.sem = platform_sem_create(0, max_count);  // 初始0个信号
    if (!g_recv_queue.sem) {
        agent_log_error("[%s] 创建信号量失败", __func__);
        platform_mutex_destroy(g_recv_queue.mutex);  // 回滚互斥锁
        g_recv_queue.mutex = NULL;
        return;
    }

    agent_log_info("[%s] 接收队列初始化完成（最大容量：%u）", __func__, max_count);
}

/**
 * 数据入队（内部函数，仅被内部调用）
 */
static ApiSnmpError net_queue_enqueue(const uint8* data, uint32 len, const char* src_ip, uint16 src_port) {
    if (!data || len == 0 || !src_ip || len > MAX_MSG_LEN) {
        agent_log_error("[%s] 入队失败：无效参数（len=%u, src_ip=%p）", __func__, len, src_ip);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 加锁保证线程安全（使用通用句柄）
    platform_mutex_lock(g_recv_queue.mutex);

    // 队列满则丢弃
    if (g_recv_queue.count >= g_recv_queue.max_count) {
        platform_mutex_unlock(g_recv_queue.mutex);
        agent_log_warn("[%s] 接收队列已满（当前：%u/%u），丢弃来自%s:%u的数据", 
                      __func__, g_recv_queue.count, g_recv_queue.max_count, src_ip, src_port);
        return API_SNMP_ERR_RESOURCE;
    }

    // 分配内部节点内存
    InternalRecvNode* node = (InternalRecvNode*)malloc(sizeof(InternalRecvNode));
    if (!node) {
        platform_mutex_unlock(g_recv_queue.mutex);
        agent_log_error("[%s] 内存不足，无法分配队列节点", __func__);
        return API_SNMP_ERR_MEMORY;
    }

    // 填充节点数据
    memcpy(node->data, data, len);
    node->len = len;
    platform_strcpy_s(node->src_ip, API_IP_LEN, src_ip);
    node->src_port = src_port;
    node->next = NULL;

    // 插入队列尾部
    if (!g_recv_queue.head) {  // 队列为空时
        g_recv_queue.head = node;
        g_recv_queue.tail = node;
    } else {  // 队列非空时
        g_recv_queue.tail->next = node;
        g_recv_queue.tail = node;
    }
    g_recv_queue.count++;

    // 解锁并发送信号量
    platform_mutex_unlock(g_recv_queue.mutex);
    platform_sem_post(g_recv_queue.sem);

    // 触发回调（如有）
    if (g_recv_callback) {
        g_recv_callback(g_callback_user_data);
    }

    agent_log_debug("[%s] 数据入队成功（来自%s:%u, 长度=%u, 队列当前大小=%u）",
                  __func__, src_ip, src_port, len, g_recv_queue.count);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 数据出队（供外部调用，仅暴露业务数据）
 */
ApiSnmpError net_interface_dequeue(NetRecvNode* node, uint32 timeout) {
    if (!node || !g_is_inited) {
        agent_log_error("[%s] 出队失败：无效参数（node=%p, 已初始化=%u）", __func__, node, g_is_inited);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 等待信号量（超时返回）
    if (!platform_sem_timedwait(g_recv_queue.sem, timeout)) {
        agent_log_debug("[%s] 出队超时（超时时间=%u ms）", __func__, timeout);
        return API_SNMP_ERR_TIMEOUT;
    }

    // 加锁操作队列
    platform_mutex_lock(g_recv_queue.mutex);

    InternalRecvNode* head = g_recv_queue.head;
    if (!head) {  // 理论上不会发生（信号量保证有数据）
        platform_mutex_unlock(g_recv_queue.mutex);
//        agent_log_warn("[%s] 信号量唤醒但队列为空（正常现象，非逻辑错误）", __func__);
        return API_SNMP_ERR_EMPTY;
    }

    // 复制数据到输出节点（隔离内部结构）
    memcpy(node->data, head->data, head->len);
    node->len = head->len;
    platform_strcpy_s(node->src_ip, API_IP_LEN, head->src_ip);
    node->src_port = head->src_port;

    // 移除头节点
    g_recv_queue.head = head->next;
    if (!g_recv_queue.head) {  // 队列空时尾指针置空
        g_recv_queue.tail = NULL;
    }
    g_recv_queue.count--;
    free(head);  // 释放节点内存

    platform_mutex_unlock(g_recv_queue.mutex);
    agent_log_debug("[%s] 数据出队成功（来自%s:%u, 长度=%u）",
                  __func__, node->src_ip, node->src_port, node->len);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 公开接口：数据入队（供底层驱动调用，替代原net_interface_on_interrupt）
 */
ApiSnmpError net_interface_enqueue(const NetRecvNode* node) {
    if (!node || !g_is_inited || node->len == 0 || node->len > MAX_MSG_LEN) {
        agent_log_error("[%s] 入队失败：无效参数（len=%u, 已初始化=%u）", 
                      __func__, node ? node->len : 0, g_is_inited);
        return API_SNMP_ERR_INVALID_PARAM;
    }
    return net_queue_enqueue(node->data, node->len, node->src_ip, node->src_port);
}

/**
 * 注册接收回调函数
 */
void net_interface_register_recv_callback(NetRecvCallback callback, void* user_data) {
    // 加锁保护回调注册（避免竞态）
    if (g_recv_queue.mutex) {  // 仅判断锁是否有效
        platform_mutex_lock(g_recv_queue.mutex);
        g_recv_callback = callback;
        g_callback_user_data = user_data;
        agent_log_debug("[%s] 接收回调注册成功（callback=%p）", __func__, callback);
        platform_mutex_unlock(g_recv_queue.mutex);
    } else {
        agent_log_warn("[%s] 互斥锁无效，回调注册失败", __func__);
    }
}

/**
 * 检查端口是否冲突（内部函数）
 */
static bool8 check_port_conflict(uint16 port) {
    for (int i = 0; i < NET_IFACE_MAX; i++) {
        if (g_registered_ifaces[i].local_port == port && g_registered_ifaces[i].net_send != NULL) {
            return BOOL8_TRUE;  // 端口已被占用
        }
    }
    return BOOL8_FALSE;
}

/**
 * 注册网络接口（供ETH/ES模块调用）
 */
ApiSnmpError net_interface_register(const NetInterface* iface) {
    if (!iface || iface->type >= NET_IFACE_MAX) {
        agent_log_error("[%s] 无效接口参数（type=%d, iface=%p）", __func__, iface ? iface->type : -1, iface);
        return API_SNMP_ERR_INVALID_PARAM;
    }
    if (!iface->net_send || !iface->net_recv) {
        agent_log_error("[%s] 接口未实现发送/接收函数（type=%d）", __func__, iface->type);
        return API_SNMP_ERR_NET_IFACE;
    }

    // 检查端口冲突（仅ETH需要）
    if (iface->local_port != 0 && check_port_conflict(iface->local_port)) {
        agent_log_error("[%s] 端口冲突：%u已被其他接口占用", __func__, iface->local_port);
        return API_SNMP_ERR_PORT_IN_USE;
    }

    // 保存接口到数组
    g_registered_ifaces[iface->type] = *iface;
    agent_log_info("[%s] 接口注册成功（type=%d, 本地端口=%u）", __func__, iface->type, iface->local_port);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 切换当前网络接口类型
 */
ApiSnmpError net_interface_set_current_type(NetInterfaceType type) {
    if (!g_is_inited || type >= NET_IFACE_MAX) {
        agent_log_error("[%s] 无效网络类型（type=%d, 已初始化=%u）", __func__, type, g_is_inited);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 检查接口是否已注册
    if (g_registered_ifaces[type].net_send == NULL || g_registered_ifaces[type].net_recv == NULL) {
        agent_log_error("[%s] 网络类型%d未注册", __func__, type);
        return API_SNMP_ERR_NET_IFACE;
    }

    // 销毁当前接口资源（如类型变更）
    if (g_current_iface && g_current_iface->destroy && type != g_current_type) {
        agent_log_debug("[%s] 销毁旧接口资源（旧type=%d）", __func__, g_current_type);
        g_current_iface->destroy();
    }

    // 更新当前接口
    g_current_type = type;
    g_current_iface = &g_registered_ifaces[type];
    agent_log_info("[%s] 已切换到网络类型：%d", __func__, type);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 获取当前网络接口类型
 */
NetInterfaceType net_interface_get_current_type(void) {
    return g_current_type;
}

/**
 * 初始化网络模块（主入口）
 */
ApiSnmpError net_interface_init(void) {
    if (g_is_inited) {
        agent_log_warn("[%s] 网络模块已初始化，无需重复操作", __func__);
        return API_SNMP_ERR_SUCCESS;
    }

    // 初始化平台层（如Winsock）
    if (!platform_init()) {
        agent_log_error("[%s] 平台初始化失败", __func__);
        return API_SNMP_ERR_INIT;
    }

    // 初始化接收队列
    net_interface_init_queue(100);
    if (!g_recv_queue.mutex || !g_recv_queue.sem) {
        agent_log_error("[%s] 接收队列初始化失败，网络模块启动终止", __func__);
        platform_deinit();
        return API_SNMP_ERR_INIT;
    }

    // 初始化ETH套接字集合（ETH专用）
    extern void eth_socket_set_init(void);
    eth_socket_set_init();
    agent_log_debug("[%s] ETH套接字集合初始化完成", __func__);

    // 初始化ETH接口（Windows平台）
    extern ApiSnmpError net_eth_win_init(void);
    ApiSnmpError err = net_eth_win_init();
    if (err != API_SNMP_ERR_SUCCESS) {
        agent_log_error("[%s] ETH接口初始化失败（错误码=%d）", __func__, err);
        net_interface_deinit();  // 回滚已初始化资源
        return err;
    }
    agent_log_info("[%s] ETH接口初始化成功", __func__);

    // 初始化ES接口（预留，失败不影响主流程）
    extern ApiSnmpError net_es_init(void);
    err = net_es_init();
    if (err != API_SNMP_ERR_SUCCESS) {
        agent_log_warn("[%s] ES接口初始化失败（非致命错误，错误码=%d）", __func__, err);
    } else {
        agent_log_info("[%s] ES接口初始化成功", __func__);
    }

    // 默认使用ETH接口
    g_current_type = NET_IFACE_ETH;
    g_current_iface = &g_registered_ifaces[NET_IFACE_ETH];
    g_is_inited = BOOL8_TRUE;
    agent_log_info("[%s] 网络模块初始化完成（当前激活接口：ETH, type=%d）", __func__, g_current_type);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 销毁网络模块（主出口）
 */
void net_interface_deinit(void) {
    if (!g_is_inited) {
        agent_log_warn("[%s] 网络模块未初始化，无需销毁", __func__);
        return;
    }

    // 销毁当前接口资源
    if (g_current_iface && g_current_iface->destroy) {
        agent_log_debug("[%s] 销毁当前接口资源（type=%d）", __func__, g_current_type);
        g_current_iface->destroy();
    }

    // 清理接收队列
    if (g_recv_queue.mutex) {
        platform_mutex_lock(g_recv_queue.mutex);
        InternalRecvNode* node = g_recv_queue.head;
        while (node) {
            InternalRecvNode* temp = node;
            node = node->next;
            free(temp);  // 释放所有节点
        }
        g_recv_queue.head = g_recv_queue.tail = NULL;
        g_recv_queue.count = 0;
        platform_mutex_unlock(g_recv_queue.mutex);

        // 销毁锁和信号量
        platform_mutex_destroy(g_recv_queue.mutex);
        g_recv_queue.mutex = NULL;
    }
    if (g_recv_queue.sem) {
        platform_sem_destroy(g_recv_queue.sem);
        g_recv_queue.sem = NULL;
    }

    // 重置全局状态
    memset(g_registered_ifaces, 0, sizeof(g_registered_ifaces));
    g_current_iface = NULL;
    g_recv_callback = NULL;
    g_callback_user_data = NULL;
    g_is_inited = BOOL8_FALSE;

    // 反初始化平台层
    platform_deinit();
    agent_log_info("[%s] 网络模块已销毁", __func__);
}

/**
 * 创建UDP套接字（ETH专用）
 */
int32 net_interface_create_udp_socket(void) {
    if (!g_is_inited || g_current_type != NET_IFACE_ETH) {
        agent_log_error("[%s] 创建失败：未初始化（%u）或非ETH模式（当前模式：%d）", 
                      __func__, g_is_inited, g_current_type);
        return -1;
    }

    // 调用平台接口创建套接字
    PlatformSocket sock = platform_udp_socket_create();
    if (sock == PLATFORM_SOCKET_INVALID) {
        agent_log_error("[%s] 创建UDP套接字失败", __func__);
        return -1;
    }

    // 添加到ETH套接字集合管理
    extern void eth_socket_set_add(int32 sock_fd);
    eth_socket_set_add((int32)sock);
    agent_log_debug("[%s] 创建UDP套接字成功（句柄：%d）", __func__, sock);
    return (int32)sock;
}

/**
 * 绑定端口（ETH专用）
 */
ApiSnmpError net_interface_bind(int32 sock_fd, uint16 port) {
    if (!g_is_inited || g_current_type != NET_IFACE_ETH || sock_fd == -1 || port == 0) {
        agent_log_error("[%s] 绑定失败：参数无效（sock=%d, port=%u, 模式=%d）", 
                      __func__, sock_fd, port, g_current_type);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 检查端口是否已占用
    if (platform_check_port_in_use(port)) {
        agent_log_error("[%s] 端口%u已被占用", __func__, port);
        return API_SNMP_ERR_PORT_IN_USE;
    }

    // 调用平台接口绑定
    if (!platform_udp_bind((PlatformSocket)sock_fd, port)) {
        agent_log_error("[%s] 套接字%d绑定端口%u失败", __func__, sock_fd, port);
        return API_SNMP_ERR_NETWORK;
    }

    agent_log_info("[%s] 套接字%d绑定端口%u成功", __func__, sock_fd, port);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 发送数据（路由到当前接口）
 */
ApiSnmpError net_interface_send_to(int32 sock_fd, const char* ip, uint16 port, const uint8* data, uint32 len) {
    if (!g_is_inited || !ip || !data || len == 0 || len > MAX_MSG_LEN) {
        agent_log_error("[%s] 发送失败：无效参数（len=%u, ip=%s, 最大长度=%d）",
                      __func__, len, ip ? ip : "NULL", MAX_MSG_LEN);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 根据当前接口类型路由
    if (g_current_type == NET_IFACE_ETH) {
        // ETH模式：直接调用平台发送接口
        int32 ret = platform_udp_sendto((PlatformSocket)sock_fd, ip, port, data, len);
        if (ret <= 0) {
            agent_log_error("[%s] ETH发送失败（sock=%d, 目标=%s:%u, 返回值=%d）",
                          __func__, sock_fd, ip, port, ret);
            return API_SNMP_ERR_NETWORK;
        }
    } else {
        // ES模式：调用注册的发送函数
        ApiSnmpError err = g_current_iface->net_send(ip, port, data, len);
        if (err != API_SNMP_ERR_SUCCESS) {
            agent_log_error("[%s] ES发送失败（type=%d, 目标=%s:%u, 错误码=%d）",
                          __func__, g_current_type, ip, port, err);
            return err;
        }
    }

    agent_log_debug("[%s] 发送成功（%u字节到%s:%u, sock=%d）", __func__, len, ip, port, sock_fd);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 接收数据（路由到当前接口）
 */
ApiSnmpError net_interface_recv_from(int32 sock_fd, uint8* buf, uint32 buf_len, char* src_ip, uint16* src_port, uint32* recv_len) {
    if (!g_is_inited || !buf || !src_ip || !src_port || !recv_len || buf_len == 0) {
        agent_log_error("[%s] 接收失败：无效参数（buf=%p, src_ip=%p）", __func__, buf, src_ip);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 根据当前接口类型路由
    if (g_current_type == NET_IFACE_ETH) {
        // ETH模式：调用平台接收接口（超时1秒）
        int32 ret = platform_udp_recvfrom((PlatformSocket)sock_fd, buf, buf_len, src_ip, src_port, 1000);
        if (ret < 0) {
            agent_log_error("[%s] ETH接收失败（sock=%d, 返回值=%d）", __func__, sock_fd, ret);
            return API_SNMP_ERR_NETWORK;
        } else if (ret == 0) {
            agent_log_debug("[%s] ETH接收超时（sock=%d, 超时时间=1000ms）", __func__, sock_fd);
            return API_SNMP_ERR_TIMEOUT;
        }
        *recv_len = ret;
    } else {
        // ES模式：调用注册的接收函数
        ApiSnmpError err = g_current_iface->net_recv(buf, buf_len, recv_len, src_ip, src_port);
        if (err != API_SNMP_ERR_SUCCESS) {
            agent_log_error("[%s] ES接收失败（type=%d, 错误码=%d）", __func__, g_current_type, err);
            return err;
        }
    }

    agent_log_debug("[%s] 接收成功（%u字节来自%s:%u, sock=%d）", __func__, *recv_len, src_ip, *src_port, sock_fd);
    return API_SNMP_ERR_SUCCESS;
}

/**
 * 关闭套接字（ETH专用）
 */
void net_interface_close(int32 sock_fd) {
    if (!g_is_inited || g_current_type != NET_IFACE_ETH || sock_fd == -1) {
        agent_log_debug("[%s] 关闭失败：无效参数（sock=%d, 模式=%d）", __func__, sock_fd, g_current_type);
        return;
    }

    // 调用平台接口关闭套接字
    platform_socket_close((PlatformSocket)sock_fd);
    // 从ETH集合中移除
    extern void eth_socket_set_remove(int32 sock_fd);
    eth_socket_set_remove(sock_fd);
    agent_log_info("[%s] 套接字%d已关闭", __func__, sock_fd);
}

/**
 * 中断数据处理（供底层驱动调用，适配新的入队接口）
 */
ApiSnmpError net_interface_on_interrupt(uint8* buf, uint32 buf_len) {
    if (!g_is_inited || !buf || buf_len == 0 || buf_len > MAX_MSG_LEN) {
        agent_log_error("[%s] 中断处理失败：无效参数（len=%u, 已初始化=%u）", 
                      __func__, buf_len, g_is_inited);
        return API_SNMP_ERR_INVALID_PARAM;
    }

    // 解析数据（调用当前接口的接收函数）
    char src_ip[API_IP_LEN] = {0};
    uint16 src_port = 0;
    uint32 recv_len = 0;
    ApiSnmpError err = g_current_iface->net_recv(buf, buf_len, &recv_len, src_ip, &src_port);
    if (err != API_SNMP_ERR_SUCCESS) {
        agent_log_error("[%s] 接口接收解析失败（错误码=%d, 源=%s:%u）", __func__, err, src_ip, src_port);
        return err;
    }

    // 数据入队（使用公开接口）
    NetRecvNode node;
    memcpy(node.data, buf, recv_len);
    node.len = recv_len;
    platform_strcpy_s(node.src_ip, API_IP_LEN, src_ip);
    node.src_port = src_port;
    return net_interface_enqueue(&node);
}
#endif
