﻿/***************************************************************************
 * File Name: nms_snmp_api.h
 * Description: NMS SNMP模块对外接口定义（供CLI及NMS APP调用）
 * Author: ZXY
 * Company: 芯祥联科技
 * Date Created: 2025-10-27
 * Version: 1.2
 * Note: 所有接口带api前缀，与核心层严格适配，依赖统一类型定义
 ***************************************************************************/
#ifndef NMS_SNMP_API_H
#define NMS_SNMP_API_H

#include "type.h"  // 引入基础基础类型（如bool8）

// 最大限制定义（API层统一约束）
#define API_MAX_AGENT_COUNT      16      // 最大Agent数量
#define SURPPORT_MAX_VARBIND_COUNT    10      // 单请求最大变量绑定数
#define API_IP_LEN               32      // IPv4/IPv6地址长度（含端口分隔符）
#define API_USERNAME_LEN         32      // V3用户名最大长度
#define API_COMMUNITY_LEN        64      // V1/V2C共同体名最大长度
#define API_OID_LEN              128     // OID字符串最大长度
#define API_VALUE_LEN            256     // 变量值最大长度
#define API_TRAP_DEFAULT_PORT    162     // 默认Trap监听端口
#define API_MAX_TRAP_VARBIND_COUNT 64    // 单Trap最大变量绑定数

// BER类型定义（遵循SNMP协议规范，修正枚举值冲突）
typedef enum {
    API_SNMP_BER_INTEGER       = 0x02,   // 整数（符合BER标准）
    API_SNMP_BER_UINTEGER32    = 0x07,   // 无符号32位整数（修正冲突，使用未占用值0x07）
    API_SNMP_BER_COUNTER32     = 0x41,   // 32位计数器（符合SNMP规范）
    API_SNMP_BER_GAUGE32       = 0x42,   // 32位计量器（符合SNMP规范）
    API_SNMP_BER_TIMETICKS     = 0x43,   // 时间ticks（1/100秒，符合规范）
    API_SNMP_BER_COUNTER64     = 0x46,   // 64位计数器（符合v2c/v3规范）
    API_SNMP_BER_IPADDRESS     = 0x0A,   // IP地址（4字节，符合规范）
    API_SNMP_BER_OCTET_STRING  = 0x04,   // 字节串（符合BER标准）
    API_SNMP_BER_OID           = 0x06,   // 对象标识符（BER标准固定值，不可修改）
    API_SNMP_BER_NULL          = 0x05    // 空值（符合BER标准）
} ApiSnmpBerType;


// SNMP版本定义
typedef enum {
    API_SNMP_VERSION_1     = 0,         // SNMPv1
    API_SNMP_VERSION_2C    = 1,         // SNMPv2c
    API_SNMP_VERSION_3     = 3,          // SNMPv3（USM安全模型）
    SNMP_VERSION_UNKNOWN   = 0xff,
} ApiSnmpVersion;

// 认证算法定义（遵循RFC 3414）
typedef enum {
    API_SNMP_AUTH_NONE     = 0,         // 无认证
	API_SNMP_AUTH_MD5 	   = 1,

    API_SNMP_AUTH_SHA1     = 2,         // SHA-1（160位）
    API_SNMP_AUTH_SHA256   = 3,          // SHA-256（256位）
	API_SNMP_AUTH_SHA256_FULL = 4,

} ApiSnmpAuthAlg;

// 加密算法定义（遵循RFC 3414）
typedef enum {
    API_SNMP_PRIV_NONE     = 0,         // 无加密
    API_SNMP_PRIV_DES      = 1,         // DES（56位密钥，RFC 3414）
    API_SNMP_PRIV_AES128   = 2,         // AES-128（128位密钥，RFC 3826）
    API_SNMP_PRIV_3DES     = 3,         // 3DES-EDE（168位密钥，补充缺失）
    API_SNMP_PRIV_AES192   = 4,         // AES-192（192位密钥，RFC 7860，补充缺失）
    API_SNMP_PRIV_AES256   = 5          // AES-256（256位密钥，RFC 7860，补充缺失）
} ApiSnmpPrivAlg;

// SNMP命令类型
typedef enum {
	API_SNMP_CMD_NONE	   = 0,  // 新增：无命令（默认状态）
    API_SNMP_CMD_GET       = 1,         // GET请求
    API_SNMP_CMD_GET_NEXT  = 2,         // GET-NEXT请求
    API_SNMP_CMD_SET       = 3,         // SET请求
    API_SNMP_CMD_TRAP      = 4,         // TRAP通知
    API_SNMP_CMD_RESPONSE  = 5,         // 响应报文
    API_SNMP_CMD_GET_BULK  = 6,          // GET-BULK请求（v2c/v3）
    API_SNMP_CMD_REPORT    = 7,  // 新增：对应 REPORT PDU（类型值7）
} ApiSnmpCmdType;

// 错误码定义（覆盖所有可能的操作结果）
typedef enum {
    API_SNMP_ERR_SUCCESS       = 0,      // 操作成功
    API_SNMP_ERR_INVALID_PARAM = 1,      // 无效参数
    API_SNMP_ERR_AGENT_NOT_FOUND = 2,    // Agent不存在
    API_SNMP_ERR_USER_NOT_FOUND = 3,     // V3用户不存在
    API_SNMP_ERR_AUTH_FAILED   = 4,      // 认证失败
    API_SNMP_ERR_PRIV_FAILED   = 5,      // 加密/解密失败
    API_SNMP_ERR_TIMEOUT       = 6,      // 操作超时
    API_SNMP_ERR_NETWORK       = 7,      // 网络错误
    API_SNMP_ERR_OID_INVALID   = 8,      // OID无效
    API_SNMP_ERR_MIB_NOT_LOADED = 9,     // MIB未加载
    API_SNMP_ERR_AGENT_EXIST   = 10,     // Agent已存在
    API_SNMP_ERR_USER_FULL     = 11,     // 用户数达上限
    API_SNMP_ERR_USER_EXIST    = 12,     // 用户已存在
    API_SNMP_ERR_MEMORY        = 13,     // 内存分配失败
    API_SNMP_ERR_INVALID_VERSION = 14,   // 无效SNMP版本
    API_SNMP_ERR_PENDING       = 15,
    API_SNMP_ERR_OP_NOT_SUPPORTED = 16,
    // 补充缺失的错误码
    API_SNMP_ERR_RESOURCE      = 17,  // 资源不足（如队列满）
    API_SNMP_ERR_EMPTY         = 18,  // 队列空
    API_SNMP_ERR_NET_IFACE     = 19,     // 网络接口错误
    API_SNMP_ERR_PORT_IN_USE   = 20,     // 端口被占用
    API_SNMP_ERR_INIT          = 21,     // 初始化失败
    API_SNMP_ERR_LOCK          = 22,     // 锁操作失败
    API_SNMP_ERR_BUFFER_TOO_SMALL = 23,  // 缓冲区太小
    API_SNMP_ERR_VERSION_MISMATCH = 24,  // 版本不匹配
    API_SNMP_ERR_ENCODE_FAILED    = 25,  // 编码失败
    API_SNMP_ERR_DECODE_FAILED    = 26,  // 解码失败

    // 新增V3安全相关错误码（解决编译错误）
    API_SNMP_ERR_TIME_OUT_OF_SYNC = 27,  // V3引擎时间同步过期
    API_SNMP_ERR_USER_DISABLED    = 28,  // V3用户被临时禁用（鉴权失败超限）
    API_SNMP_ERR_DECRYPT_FAILED   = 29,   // V3 PDU解密失败
    API_SNMP_ERR_USER_LIMIT       = 30,
    API_SNMP_ERR_ENGINE_NOT_READY = 31,
    API_SNMP_ERR_INVALID_CMD = 32,        // 新增：无效的命令类型
    API_SNMP_ERR_MISSING_ENGINE_ID = 33,  // 新增：缺少引擎ID（RFC 3414必选参数）
    API_SNMP_ERR_MISSING_ENGINE_BOOTS = 34, // 新增：缺少引擎启动次数（RFC 3414必选参数）
	API_SNMP_BER_UNKNOWN,

} ApiSnmpError;

// 日志级别定义
typedef enum {
    API_SNMP_LOG_DEBUG,                 // 调试信息
    API_SNMP_LOG_INFO,                  // 普通信息
    API_SNMP_LOG_WARN,                  // 警告信息
    API_SNMP_LOG_ERROR                  // 错误信息
} ApiSnmpLogLevel;

// 变量绑定结构（OID-值对）
typedef struct {
    char oid[API_OID_LEN];              // OID字符串（如".1.3.6.1.2.1.1.1.0"）
    uint8 type;                         // 数据类型（ApiSnmpBerType）
    union {
        int32 int_val;                  // 整数类型值
        uint32 uint_val;                // 无符号整数/计数器/计量器
        char str_val[API_VALUE_LEN];    // 字符串类型值
        uint8 bytes_val[API_VALUE_LEN]; // 字节数组类型值
        uint32 ip_val;                  // IP地址（网络字节序）
        uint64 uint64_val;              // 64位计数器
    } value;
    uint16 value_len;                   // 字节数组/字符串的实际长度
} ApiSnmpVarBind;

// 响应数据结构
typedef struct {
    uint32 request_id;                  // 对应请求的ID
    ApiSnmpError error;                 // 错误码（API层）
    int32 snmp_error_status;            // SNMP协议错误状态（如noSuchObject）
    uint8 varbind_count;                // 变量绑定数量
    ApiSnmpVarBind varbinds[SURPPORT_MAX_VARBIND_COUNT]; // 变量绑定列表
} ApiSnmpResponse;

// Agent配置结构（创建Agent时使用）
typedef struct {
    ApiSnmpVersion version;             // SNMP版本
    char community[API_COMMUNITY_LEN];  // V1/V2C共同体名（V3忽略）
} ApiSnmpAgentConfig;

// V3用户配置结构（添加V3用户时使用）
typedef struct {
    char username[API_USERNAME_LEN];    // 用户名（唯一标识）
    ApiSnmpAuthAlg auth_alg;            // 认证算法
    char* auth_pass;              // 认证密码（可为NULL，对应无认证）
    ApiSnmpPrivAlg priv_alg;            // 加密算法
    char* priv_pass;              // 加密密码（可为NULL，对应无加密）
} ApiSnmpV3UserConfig;

// Trap消息结构（接收Trap时使用）
typedef struct {
    ApiSnmpVersion version;             // SNMP版本
    union {
        char community[API_COMMUNITY_LEN]; // V1/V2C共同体名
        char username[API_USERNAME_LEN];   // V3用户名
    } sec;                               // 安全参数
    uint8 varbind_count;                // 变量绑定数量
    ApiSnmpVarBind varbinds[API_MAX_TRAP_VARBIND_COUNT]; // 变量绑定列表
} ApiSnmpTrapMessage;

// 响应回调函数类型（异步接收请求结果）
typedef void (*ApiSnmpResponseCallback)(
    char* agent_ip,               // Agent IP地址
    uint16 agent_port,                  // Agent端口
    char* username,               // V3用户名（V1/V2C为NULL）
    ApiSnmpCmdType cmd_type,            // 命令类型
    ApiSnmpResponse* response,    // 响应数据
    void* user_data                     // 用户自定义数据
);

// Trap回调函数类型（异步接收Trap消息）
typedef void (*ApiSnmpTrapCallback)(
    const char* agent_ip,               // 发送Trap的Agent IP
    uint16 agent_port,                  // 发送Trap的Agent端口
    const ApiSnmpTrapMessage* trap_msg, // Trap消息数据
    ApiSnmpError error,                 // 错误码
    void* user_data                     // 用户自定义数据
);

// -------------------------- 对外API接口声明 --------------------------

/**
 * 初始化SNMP模块（必须先调用）
 */
void api_snmp_init(void);

/**
 * 销毁SNMP模块（释放所有资源）
 */
void api_snmp_destroy(void);

/**
 * 创建SNMP Agent（管理与目标设备的连接）
 * @param ip：目标设备IP地址
 * @param port：目标设备端口（默认161）
 * @param config：Agent配置（版本、共同体名等）
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_agent_create(const char* ip, uint16 port, const ApiSnmpAgentConfig* config);

/**
 * 移除SNMP Agent（释放对应资源）
 * @param ip：目标设备IP地址
 * @param port：目标设备端口
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_agent_remove(const char* ip, uint16 port);

/**
 * 检查Agent是否存在
 * @param ip：目标设备IP地址
 * @param port：目标设备端口
 * @return 存在返回BOOL8_TRUE，否则返回BOOL8_FALSE
 */
bool8 api_snmp_agent_exist(const char* ip, uint16 port);

/**
 * 为V3版本Agent添加用户（安全认证用）
 * @param agent_ip：Agent IP地址
 * @param agent_port：Agent端口
 * @param config：V3用户配置（用户名、算法、密码等）
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_user_add(const char* agent_ip, uint16 agent_port, const ApiSnmpV3UserConfig* config);

/**
 * 移除V3版本Agent的用户
 * @param agent_ip：Agent IP地址
 * @param agent_port：Agent端口
 * @param username：用户名
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_user_remove(const char* agent_ip, uint16 agent_port, const char* username);
// 新增GET-BULK接口声明
ApiSnmpError api_snmp_get_bulk(const char* agent_ip, uint16 agent_port, const char* username,
                              uint8 non_repeaters, uint8 max_repetitions,
                              const ApiSnmpVarBind* varbinds, uint8 varbind_count,
                              uint32 timeout, void* user_data);

/**
 * 发送SNMP GET请求（获取变量值）
 * @param agent_ip：Agent IP地址
 * @param agent_port：Agent端口
 * @param username：V3用户名（V1/V2C传NULL）
 * @param varbinds：变量绑定列表（OID集合）
 * @param varbind_count：变量绑定数量
 * @param timeout：超时时间（毫秒）
 * @param user_data：用户自定义数据（回调时带回）
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_get(const char* agent_ip, uint16 agent_port, const char* username,
                         const ApiSnmpVarBind* varbinds, uint8 varbind_count,
                         uint32 timeout, void* user_data);
ApiSnmpError api_snmp_get_next(const char* agent_ip, uint16 agent_port, const char* username,
                             const ApiSnmpVarBind* varbinds, uint8 varbind_count,
                             uint32 timeout, void* user_data);

/**
 * 发送SNMP SET请求（设置变量值）
 * @param 参数同api_snmp_get，varbinds需包含值信息
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_set(const char* agent_ip, uint16 agent_port, const char* username,
                         const ApiSnmpVarBind* varbinds, uint8 varbind_count,
                         uint32 timeout, void* user_data);

/**
 * 启动Trap监听器（接收设备主动发送的Trap消息）
 * @param port：监听端口（默认162）
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_start_trap_listener(uint16 port);

/**
 * 停止Trap监听器
 */
void api_snmp_stop_trap_listener(void);

/**
 * 注册请求响应回调函数（接收GET/SET等请求的结果）
 * @param callback：回调函数指针
 */
void api_snmp_register_response_callback(ApiSnmpResponseCallback callback);

/**
 * 注册Trap回调函数（接收Trap消息）
 * @param callback：回调函数指针
 * @param user_data：用户自定义数据（回调时带回）
 */
void api_snmp_register_trap_callback(ApiSnmpTrapCallback callback, void* user_data);

/**
 * 设置日志级别（控制日志输出详细程度）
 * @param level：日志级别
 */
void api_snmp_set_log_level(ApiSnmpLogLevel level);

/**
 * 加载MIB文件（用于OID名称与数字转换）
 * @param mib_path：MIB文件路径
 * @return 成功返回API_SNMP_ERR_SUCCESS，失败返回对应错误码
 */
ApiSnmpError api_snmp_mib_load(const char* mib_path);

bool8 api_snmp_oid_validate(const char* oid);

/**
 * 构造GET/GET-NEXT请求的变量绑定（无需值）
 * @param oid_str OID字符串（如"1.3.6.1.2.1.1.1.0"）
 * @param vb 输出参数：构造好的变量绑定
 * @return 成功返回API_SNMP_ERR_SUCCESS
 */
ApiSnmpError api_snmp_create_get_varbind(const char* oid_str, ApiSnmpVarBind* vb);

/**
 * 构造SET请求的变量绑定（需类型和值）
 * @param oid_str OID字符串
 * @param type_str 类型字符串（如"int"、"string"）
 * @param value_str 值字符串
 * @param vb 输出参数：构造好的变量绑定
 * @return 成功返回API_SNMP_ERR_SUCCESS
 */
ApiSnmpError api_snmp_create_set_varbind(const char* oid_str, const char* type_str, 
                                        const char* value_str, ApiSnmpVarBind* vb);
/**
 * 将SNMP协议错误状态码（snmp_error_status）转换为可读字符串
 * @param status：SNMP协议错误状态码（如16=noSuchInstance）
 * @return 错误描述字符串（静态内存，无需释放）
 */
const char* api_snmp_error_status_to_str(int32 status);

/**
 * 将API层错误码（ApiSnmpError）转换为可读字符串
 * @param err：API层错误码（如API_SNMP_ERR_TIMEOUT）
 * @return 错误描述字符串（静态内存，无需释放）
 */
const char* api_snmp_error_to_str(ApiSnmpError err);


#endif  // NMS_SNMP_API_H
