﻿#ifndef PLATFORM_H
#define PLATFORM_H

#include "type.h"  // 基础类型（bool8、uint32等）

#define PLATFORM_THREAD_INVALID NULL
#define PLATFORM_SEM_INVALID NULL
#define PLATFORM_TIMER_INVALID NULL
#define PLATFORM_MUTEX_INVALID NULL

// -------------------------- 类型定义 --------------------------
// 线程类型与函数指针
typedef void* PlatformThread;
typedef void (*PlatformThreadFunc)(void*);

// 互斥锁类型
typedef void* PlatformMutex;

// 信号量类型（通用句柄）
typedef void* PlatformSem;

// 套接字类型
typedef int32 PlatformSocket;
#define PLATFORM_SOCKET_INVALID -1

// 定时器类型与函数指针
typedef void* PlatformTimer;
typedef void (*PlatformTimerFunc)(void*);  // 修正函数指针命名

// 时间类型（统一用64位表示时间戳）
typedef uint64 PlatformTime;

// 内存分配标记
typedef enum {
    PLATFORM_MEM_DEFAULT,    // 默认分配
    PLATFORM_MEM_SECURE      // 安全内存
} PlatformMemType;

// -------------------------- 通用函数声明（跨平台接口） --------------------------
// 线程接口
PlatformThread platform_thread_create(PlatformThreadFunc func, void* arg);
void platform_thread_join(PlatformThread thread);
void platform_thread_detach(PlatformThread thread);
uint32 platform_thread_id(void);

// 互斥锁接口
PlatformMutex platform_mutex_create(void);
void platform_mutex_lock(PlatformMutex mutex);
void platform_mutex_unlock(PlatformMutex mutex);
void platform_mutex_destroy(PlatformMutex mutex);
uint32 platform_strlen(const char* str);

// 信号量接口
PlatformSem platform_sem_create(int initial_count, int max_count);
void platform_sem_destroy(PlatformSem sem);
int32 platform_sem_post(PlatformSem sem);
int32 platform_sem_timedwait(PlatformSem sem, uint32 timeout_ms);
#define platform_sem_wait(sem) platform_sem_timedwait(sem, (uint32)-1)

// 平台初始化/反初始化
bool8 platform_init(void);
void platform_deinit(void);

// 网络接口（UDP）
PlatformSocket platform_udp_socket_create(void);
bool8 platform_udp_bind(PlatformSocket sock, uint16 port);
int32 platform_udp_sendto(PlatformSocket sock, const char* ip, uint16 port,
                         const uint8* data, uint32 len);
int32 platform_udp_recvfrom(PlatformSocket sock, uint8* buf, uint32 max_len,
                           char* src_ip, uint16* src_port, uint32 timeout_ms);
void platform_socket_close(PlatformSocket sock);
bool8 platform_check_port_in_use(uint16 port);

// 时间接口
uint32 platform_get_tick_ms(void);
uint64 platform_get_tick_us(void);
PlatformTime platform_get_utc_seconds(void);
PlatformTime platform_get_utc_ms(void);
void platform_delay_ms(uint32 ms);

// 定时器接口
PlatformTimer platform_timer_create(PlatformTimerFunc func, void* arg, uint32 period_ms);
bool8 platform_timer_start(PlatformTimer timer);
bool8 platform_timer_stop(PlatformTimer timer);
void platform_timer_destroy(PlatformTimer timer);

// 内存接口
void* platform_malloc(uint32 size);
void* platform_calloc(uint32 num, uint32 size);
//void* platform_malloc_aligned(uint32 size, uint32 align);

void platform_free(void* ptr);
void platform_free_aligned(void* ptr);

// 随机数接口
bool8 platform_rand_generate(uint8* buf, uint32 len);
bool8 platform_rand_generate_crypto(uint8* buf, uint32 len);

// 文件系统接口
uint32 platform_get_runtime_filename(char* buf, uint32 buf_len);
bool8 platform_file_exists(const char* path);
uint32 platform_get_file_size(const char* path);

// 字符串/内存操作
uint32 platform_memcpy_s(void* dest, uint32 dest_max_len, const void* src, uint32 src_len);
uint32 platform_memset_s(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len);
uint32 platform_strcpy_s(char* dest, uint32 dest_max_len, const char* src);
int32 platform_strcmp(const char* str1, const char* str2);
char* platform_strchr(const char* str, int32 ch);
uint32 platform_be32_to_cpu(const uint8* in);
void platform_cpu_to_be32(uint8* out, uint32 value);

// -------------------------- Windows 平台特有实现声明（仅补充） --------------------------
#ifdef WIN32
// 线程实现
uint32 platform_thread_id_win32(void);
PlatformThread platform_thread_create_win32(PlatformThreadFunc func, void* arg);
void platform_thread_join_win32(PlatformThread thread);
void platform_thread_detach_win32(PlatformThread thread);

// 互斥锁实现
PlatformMutex platform_mutex_create_win32(void);
void platform_mutex_lock_win32(PlatformMutex mutex);
void platform_mutex_unlock_win32(PlatformMutex mutex);
void platform_mutex_destroy_win32(PlatformMutex mutex);
uint32 platform_strlen_win32(const char* str);

// 信号量实现
PlatformSem platform_sem_create_win32(int initial_count, int max_count);
void platform_sem_destroy_win32(PlatformSem sem);
int32 platform_sem_post_win32(PlatformSem sem);
int32 platform_sem_timedwait_win32(PlatformSem sem, uint32 timeout_ms);

// 平台初始化实现
bool8 platform_init_win32(void);
void platform_deinit_win32(void);

// 网络接口实现
PlatformSocket platform_udp_socket_create_win32(void);
bool8 platform_udp_bind_win32(PlatformSocket sock, uint16 port);
int32 platform_udp_sendto_win32(PlatformSocket sock, const char* ip, uint16 port,
                               const uint8* data, uint32 len);
int32 platform_udp_recvfrom_win32(PlatformSocket sock, uint8* buf, uint32 max_len,
                                char* src_ip, uint16* src_port, uint32 timeout_ms);
void platform_socket_close_win32(PlatformSocket sock);
bool8 platform_check_port_in_use_win32(uint16 port);

// 时间接口实现
uint32 platform_get_tick_ms_win32(void);
uint64 platform_get_tick_us_win32(void);
PlatformTime platform_get_utc_seconds_win32(void);
PlatformTime platform_get_utc_ms_win32(void);
void platform_delay_ms_win32(uint32 ms);

// 定时器实现（核心：明确声明避免类型截断）
PlatformTimer platform_timer_create_win32(PlatformTimerFunc func, void* arg, uint32 period_ms);
bool8 platform_timer_start_win32(PlatformTimer timer);
bool8 platform_timer_stop_win32(PlatformTimer timer);
void platform_timer_destroy_win32(PlatformTimer timer);

// 内存接口实现
void* platform_malloc_win32(uint32 size);
void* platform_calloc_win32(uint32 num, uint32 size);
void* platform_malloc_aligned_win32(uint32 size, uint32 align, PlatformMemType type);
void platform_free_win32(void* ptr);
void platform_free_aligned_win32(void* ptr);

// 随机数接口实现
bool8 platform_rand_generate_win32(uint8* buf, uint32 len);
bool8 platform_rand_generate_crypto_win32(uint8* buf, uint32 len);

// 文件系统接口实现
bool8 platform_file_exists_win32(const char* path);
uint32 platform_get_file_size_win32(const char* path);
uint32 platform_get_runtime_filename_win32(char* buf, uint32 buf_len);

// 字符串/内存操作实现
uint32 platform_memcpy_s_win32(void* dest, uint32 dest_max_len, const void* src, uint32 src_len);
uint32 platform_memset_s_win32(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len);
uint32 platform_strcpy_s_win32(char* dest, uint32 dest_max_len, const char* src);
int32 platform_strcmp_win32(const char* str1, const char* str2);
char* platform_strchr_win32(const char* str, int32 ch);
#endif  // WIN32

#endif  // PLATFORM_H
