﻿/***************************************************************************
 * File Name: platform_linux.c
 * Description: Linux平台具体实现（基于POSIX接口）
 * Author: ZXY
 * Company: 芯祥联科技
 * Date Created: 2025-10-22
 * Version: 1.2（修复recvfrom拼写，补充端口检查）
 ***************************************************************************/
#ifdef LINUX
#include "type.h"
#include "platform.h"
#include "agent_log.h"
#include <pthread.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <timerfd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <string.h>
#include <limits.h>
#include <time.h>
#include <sys/stat.h>
#include <semaphore.h>
#include <stdarg.h>

// 定时器上下文结构（扩展状态控制）
typedef struct {
    int32_t fd;                // timerfd文件描述符
    int32_t wakeup_pipe[2];     // 唤醒管道（[0]读，[1]写）
    PlatformTimerFunc func;    // 回调函数
    void* arg;                 // 用户参数
    volatile bool8 is_stop;    // 线程停止标志
    volatile bool8 is_running; // 定时器运行状态
    pthread_t tid;             // 定时器线程ID
    uint32 period_ms;          // 周期（毫秒）
} TimerCtx;

// -------------------------- 线程实现（补充线程ID） --------------------------
uint32 platform_thread_id_linux(void) {
    return (uint32)pthread_self();  // 返回当前线程ID（POSIX线程ID）
}

PlatformThread platform_thread_create_linux(PlatformThreadFunc func, void* arg) {
    pthread_t* tid = (pthread_t*)malloc(sizeof(pthread_t));
    if (tid == NULL) {
        agent_log_error("Failed to allocate thread ID");
        return NULL;
    }
    if (pthread_create(tid, NULL, func, arg) != 0) {
        agent_log_error("pthread_create failed: %s", strerror(errno));
        free(tid);
        return NULL;
    }
    return (PlatformThread)tid;
}

void platform_thread_join_linux(PlatformThread thread) {
    if (thread != NULL) {
        pthread_t* tid = (pthread_t*)thread;
        pthread_join(*tid, NULL);
        free(tid);
    }
}

void platform_thread_detach_linux(PlatformThread thread) {
    if (thread != NULL) {
        pthread_t* tid = (pthread_t*)thread;
        pthread_detach(*tid);
        free(tid);
    }
}

// -------------------------- 互斥锁实现（保持不变） --------------------------
PlatformMutex platform_mutex_create_linux(void) {
    pthread_mutex_t* mutex = (pthread_mutex_t*)malloc(sizeof(pthread_mutex_t));
    if (mutex == NULL) {
        agent_log_error("Failed to allocate mutex");
        return NULL;
    }
    if (pthread_mutex_init(mutex, NULL) != 0) {
        agent_log_error("pthread_mutex_init failed: %s", strerror(errno));
        free(mutex);
        return NULL;
    }
    return (PlatformMutex)mutex;
}

void platform_mutex_lock_linux(PlatformMutex mutex) {
    if (mutex != NULL) {
        pthread_mutex_lock((pthread_mutex_t*)mutex);
    }
}

void platform_mutex_unlock_linux(PlatformMutex mutex) {
    if (mutex != NULL) {
        pthread_mutex_unlock((pthread_mutex_t*)mutex);
    }
}

void platform_mutex_destroy_linux(PlatformMutex mutex) {
    if (mutex != NULL) {
        pthread_mutex_destroy((pthread_mutex_t*)mutex);
        free(mutex);
    }
}

// -------------------------- 信号量实现（保持不变） --------------------------
PlatformSem platform_sem_create_linux(int initial_count, int max_count) {
    sem_t* sem = (sem_t*)malloc(sizeof(sem_t));
    if (sem == NULL) {
        agent_log_error("Failed to allocate semaphore");
        return NULL;
    }
    if (sem_init(sem, 0, initial_count) != 0) {
        agent_log_error("sem_init failed: %s", strerror(errno));
        free(sem);
        return NULL;
    }
    return (PlatformSem)sem;
}

void platform_sem_destroy_linux(PlatformSem sem) {
    if (sem != NULL) {
        sem_destroy((sem_t*)sem);
        free(sem);
    }
}

int32 platform_sem_post_linux(PlatformSem sem) {
    return sem_post((sem_t*)sem) == 0 ? 0 : -1;
}

int32 platform_sem_timedwait_linux(PlatformSem sem, uint32 timeout_ms) {
    struct timespec ts;
    clock_gettime(CLOCK_REALTIME, &ts);
    ts.tv_sec += timeout_ms / 1000;
    ts.tv_nsec += (timeout_ms % 1000) * 1000000;
    if (ts.tv_nsec >= 1000000000) {
        ts.tv_sec++;
        ts.tv_nsec -= 1000000000;
    }
    int ret = sem_timedwait((sem_t*)sem, &ts);
    if (ret == 0) return 0;
    return (errno == ETIMEDOUT) ? -2 : -1;
}

// -------------------------- 网络实现（补充错误日志） --------------------------
bool8 platform_init_linux(void) {
    return BOOL8_TRUE;  // Linux无需初始化网络库
}

PlatformSocket platform_udp_socket_create_linux(void) {
    int32_t sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock == -1) {
        agent_log_error("socket creation failed: %s", strerror(errno));
        return PLATFORM_SOCKET_INVALID;
    }
    return (PlatformSocket)sock;
}

bool8 platform_udp_bind_linux(PlatformSocket sock, uint16 port) {
    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port);

    if (bind((int32_t)sock, (struct sockaddr*)&addr, sizeof(addr)) != 0) {
        agent_log_error("bind failed (port %d): %s", port, strerror(errno));
        return BOOL8_FALSE;
    }
    return BOOL8_TRUE;
}

int32 platform_udp_sendto_linux(PlatformSocket sock, const char* ip, uint16 port,
                               const uint8* data, uint32 len) {
    if (sock == PLATFORM_SOCKET_INVALID || ip == NULL || data == NULL || len == 0) {
        return -1;
    }

    struct sockaddr_in dest_addr;
    memset(&dest_addr, 0, sizeof(dest_addr));
    dest_addr.sin_family = AF_INET;
    dest_addr.sin_port = htons(port);

    if (inet_pton(AF_INET, ip, &dest_addr.sin_addr) != 1) {
        agent_log_error("Invalid IP address: %s", ip);
        return -1;  // IP解析失败
    }

    ssize_t ret = sendto((int32_t)sock, data, len, 0,
                        (struct sockaddr*)&dest_addr, sizeof(dest_addr));
    return (ret == -1) ? -1 : (int32)ret;
}

int32 platform_udp_recvfrom_linux(PlatformSocket sock, uint8* buf, uint32 max_len,
                                char* src_ip, uint16* src_port, uint32 timeout_ms) {
    if (sock == PLATFORM_SOCKET_INVALID || buf == NULL || max_len == 0) {
        return -1;
    }

    struct sockaddr_in src_addr;
    socklen_t addr_len = sizeof(src_addr);
    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET((int32_t)sock, &read_fds);

    struct timeval tv;
    tv.tv_sec = timeout_ms / 1000;
    tv.tv_usec = (timeout_ms % 1000) * 1000;

    int32_t ret = select((int32_t)sock + 1, &read_fds, NULL, NULL, &tv);
    if (ret == 0) {
        return 0;  // 超时
    } else if (ret < 0) {
        agent_log_error("select failed: %s", strerror(errno));
        return -1;  // 错误
    }

    ret = recvfrom((int32_t)sock, buf, max_len, 0,
                  (struct sockaddr*)&src_addr, &addr_len);
    if (ret <= 0) {
        agent_log_error("recvfrom failed: %s", strerror(errno));
        return -1;  // 接收失败
    }

    if (src_ip != NULL) {
        inet_ntop(AF_INET, &src_addr.sin_addr, src_ip, 16);
        src_ip[15] = '\0';
    }
    if (src_port != NULL) {
        *src_port = ntohs(src_addr.sin_port);
    }

    return ret;
}

void platform_socket_close_linux(PlatformSocket sock) {
    if (sock != PLATFORM_SOCKET_INVALID) {
        close((int32_t)sock);
    }
}

bool8 platform_check_port_in_use_linux(uint16 port) {
    int32_t temp_sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (temp_sock == -1) {
        agent_log_error("socket creation failed: %s", strerror(errno));
        return BOOL8_TRUE;
    }

    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port);

    bool8 in_use = BOOL8_TRUE;
    if (bind(temp_sock, (struct sockaddr*)&addr, sizeof(addr)) == 0) {
        in_use = BOOL8_FALSE;
    }
    close(temp_sock);
    return in_use;
}

// -------------------------- 时间实现（补充UTC时间） --------------------------
uint32 platform_get_tick_ms_linux(void) {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return (uint32)(tv.tv_sec * 1000 + tv.tv_usec / 1000);
}

uint64 platform_get_tick_us_linux(void) {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return (uint64)tv.tv_sec * 1000000 + tv.tv_usec;
}

PlatformTime platform_get_utc_seconds_linux(void) {
    time_t utc_time;
    time(&utc_time);  // 获取当前UTC时间（秒级）
    return (PlatformTime)utc_time;
}

PlatformTime platform_get_utc_ms_linux(void) {
    struct timespec ts;
    clock_gettime(CLOCK_REALTIME, &ts);  // 高精度UTC时间
    return (PlatformTime)(ts.tv_sec * 1000ULL + ts.tv_nsec / 1000000);
}

void platform_delay_ms_linux(uint32 ms) {
    if (ms == 0) return;
    usleep(ms * 1000);
}

// -------------------------- 定时器实现（补充启停控制） --------------------------
static void* timer_thread_func(void* arg) {
    TimerCtx* ctx = (TimerCtx*)arg;
    uint64_t exp;
    fd_set read_fds;

    while (!ctx->is_stop) {
        FD_ZERO(&read_fds);
        FD_SET(ctx->fd, &read_fds);
        FD_SET(ctx->wakeup_pipe[0], &read_fds);
        int32_t max_fd = (ctx->fd > ctx->wakeup_pipe[0]) ? ctx->fd : ctx->wakeup_pipe[0];

        int32_t ret = select(max_fd + 1, &read_fds, NULL, NULL, NULL);
        if (ret == -1 && errno != EINTR) {
            agent_log_error("timer select failed: %s", strerror(errno));
            break;
        }

        if (ctx->is_stop) break;

        if (FD_ISSET(ctx->fd, &read_fds)) {
            read(ctx->fd, &exp, sizeof(exp));  // 清除超时标志
            if (ctx->func && ctx->is_running) {
                ctx->func(ctx->arg);  // 仅在运行状态调用回调
            }
        }
    }

    close(ctx->wakeup_pipe[0]);
    close(ctx->wakeup_pipe[1]);
    return NULL;
}

PlatformTimer platform_timer_create_linux(PlatformTimerFunc func, void* arg, uint32 period_ms) {
    if (func == NULL || period_ms == 0) return NULL;

    // 创建唤醒管道
    TimerCtx* ctx = (TimerCtx*)malloc(sizeof(TimerCtx));
    if (!ctx) {
        agent_log_error("Failed to allocate timer context");
        return NULL;
    }
    if (pipe(ctx->wakeup_pipe) == -1) {
        agent_log_error("pipe creation failed: %s", strerror(errno));
        free(ctx);
        return NULL;
    }

    // 创建timerfd
    ctx->fd = timerfd_create(CLOCK_MONOTONIC, TFD_NONBLOCK);
    if (ctx->fd == -1) {
        agent_log_error("timerfd_create failed: %s", strerror(errno));
        close(ctx->wakeup_pipe[0]);
        close(ctx->wakeup_pipe[1]);
        free(ctx);
        return NULL;
    }

    // 保存周期参数（用于启停控制）
    ctx->period_ms = period_ms;
    ctx->func = func;
    ctx->arg = arg;
    ctx->is_stop = BOOL8_FALSE;
    ctx->is_running = BOOL8_TRUE;  // 创建后默认运行

    // 配置定时器
    struct itimerspec its;
    its.it_interval.tv_sec = period_ms / 1000;
    its.it_interval.tv_nsec = (period_ms % 1000) * 1000000;
    its.it_value = its.it_interval;  // 立即启动
    if (timerfd_settime(ctx->fd, 0, &its, NULL) == -1) {
        agent_log_error("timerfd_settime failed: %s", strerror(errno));
        close(ctx->fd);
        close(ctx->wakeup_pipe[0]);
        close(ctx->wakeup_pipe[1]);
        free(ctx);
        return NULL;
    }

    // 创建线程
    if (pthread_create(&ctx->tid, NULL, timer_thread_func, ctx) != 0) {
        agent_log_error("pthread_create failed: %s", strerror(errno));
        close(ctx->fd);
        close(ctx->wakeup_pipe[0]);
        close(ctx->wakeup_pipe[1]);
        free(ctx);
        return NULL;
    }

    return (PlatformTimer)ctx;
}

bool8 platform_timer_start_linux(PlatformTimer timer) {
    if (!timer) return BOOL8_FALSE;
    TimerCtx* ctx = (TimerCtx*)timer;

    // 配置定时器周期
    struct itimerspec its;
    its.it_interval.tv_sec = ctx->period_ms / 1000;
    its.it_interval.tv_nsec = (ctx->period_ms % 1000) * 1000000;
    its.it_value = its.it_interval;  // 立即启动

    if (timerfd_settime(ctx->fd, 0, &its, NULL) == -1) {
        agent_log_error("timerfd_settime failed: %s", strerror(errno));
        return BOOL8_FALSE;
    }

    ctx->is_running = BOOL8_TRUE;
    return BOOL8_TRUE;
}

bool8 platform_timer_stop_linux(PlatformTimer timer) {
    if (!timer) return BOOL8_FALSE;
    TimerCtx* ctx = (TimerCtx*)timer;

    // 停止定时器（设置it_value为0）
    struct itimerspec its = {0};
    if (timerfd_settime(ctx->fd, 0, &its, NULL) == -1) {
        agent_log_error("timerfd_settime failed: %s", strerror(errno));
        return BOOL8_FALSE;
    }

    ctx->is_running = BOOL8_FALSE;
    return BOOL8_TRUE;
}

void platform_timer_destroy_linux(PlatformTimer timer) {
    if (timer != NULL) {
        TimerCtx* ctx = (TimerCtx*)timer;
        ctx->is_stop = BOOL8_TRUE;
        ctx->is_running = BOOL8_FALSE;
        
        uint8_t dummy = 0;
        write(ctx->wakeup_pipe[1], &dummy, 1);  // 唤醒线程
        pthread_join(ctx->tid, NULL);           // 等待线程退出
        close(ctx->fd);
        free(ctx);
    }
}

// -------------------------- 内存接口实现（补充基础分配） --------------------------
void* platform_malloc_linux(uint32 size) {
    return malloc(size);
}

void* platform_calloc_linux(uint32 num, uint32 size) {
    return calloc(num, size);
}

void platform_free_linux(void* ptr) {
    if (ptr) free(ptr);
}

void* platform_malloc_aligned_linux(uint32 size, uint32 align) {
    if (size == 0 || align == 0 || (align & (align - 1)) != 0 || align < sizeof(void*)) {
        agent_log_error("Invalid alignment parameters");
        return NULL;
    }
    size = (size + align - 1) & ~(align - 1);  // 确保size是align的倍数
    return aligned_alloc(align, size);
}

void platform_free_aligned_linux(void* ptr) {
    if (ptr != NULL) {
        free(ptr);
    }
}

// -------------------------- 随机数实现（补充加密级） --------------------------
bool8 platform_rand_generate_linux(uint8* buf, uint32 len) {
    if (buf == NULL || len == 0) return BOOL8_FALSE;

    // 普通随机数使用标准库（非加密级）
    for (uint32 i = 0; i < len; i++) {
        buf[i] = (uint8)(rand() % 256);
    }
    return BOOL8_TRUE;
}

bool8 platform_rand_generate_crypto_linux(uint8* buf, uint32 len) {
    if (buf == NULL || len == 0) return BOOL8_FALSE;

    // 加密级随机数使用/dev/random（阻塞型，适合密钥生成）
    int32_t fd = open("/dev/random", O_RDONLY);
    if (fd == -1) {
        agent_log_error("Failed to open /dev/random: %s", strerror(errno));
        return BOOL8_FALSE;
    }

    ssize_t read_len = 0;
    while (read_len < (ssize_t)len) {
        ssize_t ret = read(fd, buf + read_len, len - read_len);
        if (ret == -1) {
            if (errno != EINTR) {
                agent_log_error("read /dev/random failed: %s", strerror(errno));
                close(fd);
                return BOOL8_FALSE;
            }
        } else {
            read_len += ret;
        }
    }

    close(fd);
    return BOOL8_TRUE;
}

// -------------------------- 文件系统接口实现 --------------------------
bool8 platform_file_exists_linux(const char* path) {
    if (!path) return BOOL8_FALSE;
    struct stat st;
    return (stat(path, &st) == 0) ? BOOL8_TRUE : BOOL8_FALSE;
}

uint32 platform_get_file_size_linux(const char* path) {
    if (!path) return 0;
    struct stat st;
    if (stat(path, &st) != 0) return 0;
    return (uint32)st.st_size;  // 适合小文件（<4GB）
}

// -------------------------- 字符串/内存操作实现 --------------------------
uint32 platform_memset_s_linux(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len) {
    if (!dest || dest_max_len == 0) return 0;
    uint32 actual_len = (set_len < dest_max_len) ? set_len : dest_max_len;
    memset(dest, value, actual_len);
    return actual_len;
}

int32 platform_strcmp_linux(const char* str1, const char* str2) {
    return strcmp(str1, str2);
}

char* platform_strchr_linux(const char* str, int32 ch) {
    return strchr(str, ch);
}

// -------------------------- 日志接口实现 --------------------------
void platform_log_output_linux(const char* module, const char* fmt, va_list args) {
    char log_buf[512] = {0};
    vsnprintf(log_buf, sizeof(log_buf), fmt, args);
    agent_log_info(module, "%s", log_buf);
}

// -------------------------- 运行时文件名接口（保持不变） --------------------------
uint32 platform_get_runtime_filename_linux(char* buf, uint32 buf_len) {
    if (buf == NULL || buf_len == 0) return 0;

    char full_path[PATH_MAX] = {0};
    ssize_t path_len = readlink("/proc/self/exe", full_path, sizeof(full_path) - 1);
    if (path_len == -1) {
        agent_log_error("readlink failed: %s", strerror(errno));
        return 0;
    }

    char* filename = strrchr(full_path, '/');
    filename = (filename == NULL) ? full_path : (filename + 1);

    uint32 copy_len = strlen(filename);
    if (copy_len >= buf_len) copy_len = buf_len - 1;
    strncpy(buf, filename, copy_len);
    buf[copy_len] = '\0';

    return copy_len;
}

// -------------------------- 平台反初始化 --------------------------
void platform_deinit_linux(void) {
    // Linux无需额外清理
}
#endif
