﻿/***************************************************************************
 * File Name: platform_stm32.c
 * Description: STM32F4平台实现（FreeRTOS+LWIP）
 * Author: ZXY
 * Company: 芯祥联科技
 * Date Created: 2025-10-22
 * Version: 1.2（修复recv拼写，补充端口检查）
 ***************************************************************************/
#ifdef STM32
#include "type.h"
#include "platform.h"
#include "agent_log.h"
#include "stm32f4xx_hal.h"
#include "lwip/udp.h"
#include "lwip/inet.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "timers.h"
#include <string.h>

// 全局硬件句柄
static TIM_HandleTypeDef htim7;  // 微秒级定时器（TIM7）
static TIM_HandleTypeDef htim2;  // UTC时间基准定时器（TIM2）
static RNG_HandleTypeDef hrng;   // 硬件随机数发生器

// UDP接收缓冲区结构（每个套接字独立）
typedef struct {
    uint8* buf;         // 接收缓冲区
    uint32 max_len;     // 缓冲区最大长度
    uint32 recv_len;    // 实际接收长度
    char src_ip[16];    // 源IP地址
    uint16 src_port;    // 源端口
    SemaphoreHandle_t sem; // 同步信号量
} UdpRecvCtx;

// UDP套接字上下文（绑定控制块和接收上下文）
typedef struct {
    struct udp_pcb* pcb;    // LWIP UDP控制块
    UdpRecvCtx recv_ctx;    // 接收上下文
} UdpSocketCtx;

// 定时器扩展结构（支持动态启停）
typedef struct {
    TimerHandle_t timer;    // FreeRTOS软件定时器
    PlatformTimerFunc func; // 回调函数
    void* arg;              // 用户参数
    uint32 period_ms;       // 周期（毫秒）
    bool8 is_running;       // 运行状态
} Stm32Timer;

// UTC时间基准（秒级，通过NTP同步更新）
static volatile PlatformTime g_utc_seconds = 0;

// -------------------------- 线程实现（FreeRTOS） --------------------------
uint32 platform_thread_id_stm32(void) {
    return (uint32)xTaskGetCurrentTaskHandle();  // 返回任务句柄作为ID
}

PlatformThread platform_thread_create_stm32(PlatformThreadFunc func, void* arg) {
    TaskHandle_t* handle = (TaskHandle_t*)pvPortMalloc(sizeof(TaskHandle_t));
    if (handle == NULL) {
        agent_log_error("Failed to allocate thread handle");
        return NULL;
    }

    // 创建任务（栈大小1024字，优先级5）
    BaseType_t ret = xTaskCreate(
        (TaskFunction_t)func, 
        "STM32_Thread", 
        1024, 
        arg, 
        5, 
        handle
    );
    if (ret != pdPASS) {
        agent_log_error("xTaskCreate failed: %d", ret);
        vPortFree(handle);
        return NULL;
    }
    return (PlatformThread)handle;
}

void platform_thread_join_stm32(PlatformThread thread) {
    if (thread != NULL) {
        TaskHandle_t* handle = (TaskHandle_t*)thread;
        if (eTaskGetState(*handle) != eDeleted) {
            vTaskDelete(*handle);  // FreeRTOS无等待join，直接删除
        }
        vPortFree(thread);
    }
}

void platform_thread_detach_stm32(PlatformThread thread) {
    if (thread != NULL) {
        vPortFree(thread);  // FreeRTOS无分离概念，仅释放句柄
    }
}

// -------------------------- 互斥锁实现（FreeRTOS） --------------------------
PlatformMutex platform_mutex_create_stm32(void) {
    SemaphoreHandle_t mutex = xSemaphoreCreateMutex();
    if (!mutex) {
        agent_log_error("xSemaphoreCreateMutex failed");
    }
    return (PlatformMutex)mutex;
}

void platform_mutex_lock_stm32(PlatformMutex mutex) {
    xSemaphoreTake((SemaphoreHandle_t)mutex, portMAX_DELAY);  // 无限等待
}

void platform_mutex_unlock_stm32(PlatformMutex mutex) {
    xSemaphoreGive((SemaphoreHandle_t)mutex);
}

void platform_mutex_destroy_stm32(PlatformMutex mutex) {
    vSemaphoreDelete((SemaphoreHandle_t)mutex);
}

// -------------------------- 信号量实现（FreeRTOS） --------------------------
PlatformSem platform_sem_create_stm32(int initial_count, int max_count) {
    SemaphoreHandle_t sem = xSemaphoreCreateCounting(max_count, initial_count);
    if (!sem) {
        agent_log_error("xSemaphoreCreateCounting failed");
    }
    return (PlatformSem)sem;
}

void platform_sem_destroy_stm32(PlatformSem sem) {
    vSemaphoreDelete((SemaphoreHandle_t)sem);
}

int32 platform_sem_post_stm32(PlatformSem sem) {
    return xSemaphoreGive((SemaphoreHandle_t)sem) ? 0 : -1;
}

int32 platform_sem_timedwait_stm32(PlatformSem sem, uint32 timeout_ms) {
    BaseType_t ret = xSemaphoreTake(
        (SemaphoreHandle_t)sem, 
        pdMS_TO_TICKS(timeout_ms)
    );
    return ret == pdPASS ? 0 : -2;  // -2表示超时
}

// -------------------------- UDP接收回调（LWIP） --------------------------
static void udp_recv_callback(void* arg, struct udp_pcb* pcb, struct pbuf* p, 
                            const ip_addr_t* src_ip, u16_t src_port) {
    UdpSocketCtx* ctx = (UdpSocketCtx*)arg;
    if (p == NULL || ctx == NULL || ctx->recv_ctx.buf == NULL) {
        if (p != NULL) pbuf_free(p);
        return;
    }

    // 复制接收数据（限制最大长度）
    ctx->recv_ctx.recv_len = (p->tot_len > ctx->recv_ctx.max_len) ? 
                          ctx->recv_ctx.max_len : p->tot_len;
    pbuf_copy_partial(p, ctx->recv_ctx.buf, ctx->recv_ctx.recv_len, 0);

    // 记录源IP和端口
    inet_ntop(AF_INET, src_ip, ctx->recv_ctx.src_ip, sizeof(ctx->recv_ctx.src_ip));
    ctx->recv_ctx.src_port = src_port;

    pbuf_free(p);
    xSemaphoreGive(ctx->recv_ctx.sem);  // 通知接收完成
}

// -------------------------- 网络实现（LWIP） --------------------------
bool8 platform_init_stm32(void) {
    // 初始化HAL库
    if (HAL_Init() != HAL_OK) {
        agent_log_error("HAL_Init failed");
        return BOOL8_FALSE;
    }

    // 初始化硬件外设
    MX_TIM7_Init();   // 微秒定时器
    MX_TIM2_Init();   // UTC时间基准定时器
    MX_RNG_Init();    // 硬件随机数
    MX_LWIP_Init();   // LWIP协议栈

    return BOOL8_TRUE;
}

PlatformSocket platform_udp_socket_create_stm32(void) {
    // 分配套接字上下文
    UdpSocketCtx* ctx = (UdpSocketCtx*)pvPortMalloc(sizeof(UdpSocketCtx));
    if (ctx == NULL) {
        agent_log_error("Failed to allocate UDP context");
        return PLATFORM_SOCKET_INVALID;
    }

    // 创建UDP控制块
    ctx->pcb = udp_new();
    if (ctx->pcb == NULL) {
        agent_log_error("udp_new failed");
        vPortFree(ctx);
        return PLATFORM_SOCKET_INVALID;
    }

    // 初始化接收上下文
    ctx->recv_ctx.sem = xSemaphoreCreateBinary();
    if (ctx->recv_ctx.sem == NULL) {
        agent_log_error("xSemaphoreCreateBinary failed");
        udp_remove(ctx->pcb);
        vPortFree(ctx);
        return PLATFORM_SOCKET_INVALID;
    }
    ctx->recv_ctx.buf = NULL;
    ctx->recv_ctx.max_len = 0;
    ctx->recv_ctx.recv_len = 0;

    // 注册回调
    udp_recv(ctx->pcb, udp_recv_callback, ctx);
    return (PlatformSocket)ctx;
}

bool8 platform_udp_bind_stm32(PlatformSocket sock, uint16 port) {
    UdpSocketCtx* ctx = (UdpSocketCtx*)sock;
    if (ctx == NULL || ctx->pcb == NULL) return BOOL8_FALSE;

    return (udp_bind(ctx->pcb, IP_ADDR_ANY, port) == ERR_OK) ? BOOL8_TRUE : BOOL8_FALSE;
}

int32 platform_udp_sendto_stm32(PlatformSocket sock, const char* ip, uint16 port,
                               const uint8* data, uint32 len) {
    UdpSocketCtx* ctx = (UdpSocketCtx*)sock;
    if (ctx == NULL || ctx->pcb == NULL || ip == NULL || data == NULL || len == 0) {
        return -1;
    }

    ip_addr_t dest_ip;
    if (inet_pton(AF_INET, ip, &dest_ip.addr) != 1) {
        agent_log_error("Invalid IP address: %s", ip);
        return -1;  // IP解析失败
    }

    struct pbuf* p = pbuf_alloc(PBUF_TRANSPORT, len, PBUF_RAM);
    if (p == NULL) {
        agent_log_error("pbuf_alloc failed");
        return -1;
    }

    pbuf_take(p, data, len);
    err_t err = udp_sendto(ctx->pcb, p, &dest_ip, port);
    pbuf_free(p);

    return (err == ERR_OK) ? (int32)len : -1;
}

int32 platform_udp_recvfrom_stm32(PlatformSocket sock, uint8* buf, uint32 max_len,
                                char* src_ip, uint16* src_port, uint32 timeout_ms) {
    UdpSocketCtx* ctx = (UdpSocketCtx*)sock;
    if (ctx == NULL || ctx->pcb == NULL || buf == NULL || max_len == 0) {
        return -1;
    }

    // 配置接收缓冲区
    ctx->recv_ctx.buf = buf;
    ctx->recv_ctx.max_len = max_len;
    ctx->recv_ctx.recv_len = 0;
    xSemaphoreReset(ctx->recv_ctx.sem);

    // 等待数据或超时
    BaseType_t ret = xSemaphoreTake(
        ctx->recv_ctx.sem, 
        pdMS_TO_TICKS(timeout_ms)
    );
    if (ret != pdPASS) {
        return 0;  // 超时
    }

    // 填充源信息
    if (src_ip != NULL) {
        strncpy(src_ip, ctx->recv_ctx.src_ip, 16);
        src_ip[15] = '\0';
    }
    if (src_port != NULL) {
        *src_port = ctx->recv_ctx.src_port;
    }

    return (int32)ctx->recv_ctx.recv_len;
}

void platform_socket_close_stm32(PlatformSocket sock) {
    UdpSocketCtx* ctx = (UdpSocketCtx*)sock;
    if (ctx == NULL) return;

    if (ctx->pcb != NULL) {
        udp_remove(ctx->pcb);  // 移除UDP控制块
    }
    if (ctx->recv_ctx.sem != NULL) {
        vSemaphoreDelete(ctx->recv_ctx.sem);  // 释放信号量
    }
    vPortFree(ctx);  // 释放上下文
}

bool8 platform_check_port_in_use_stm32(uint16 port) {
    struct udp_pcb* pcb = udp_pcbs;  // 遍历LWIP UDP控制块链表
    while (pcb != NULL) {
        if (pcb->local_port == port) {
            return BOOL8_TRUE;  // 端口已占用
        }
        pcb = pcb->next;
    }
    return BOOL8_FALSE;  // 端口未占用
}

// -------------------------- 时间实现（补充UTC时间） --------------------------
uint32 platform_get_tick_ms_stm32(void) {
    return HAL_GetTick();  // 基于SysTick的毫秒计数
}

uint64 platform_get_tick_us_stm32(void) {
    return (uint64)HAL_GetTick() * 1000 + (uint64)__HAL_TIM_GET_COUNTER(&htim7);
}

// UTC秒级时间（需通过NTP或外部时钟同步）
PlatformTime platform_get_utc_seconds_stm32(void) {
    return g_utc_seconds;
}

// UTC毫秒级时间（基于TIM2和UTC秒基准）
PlatformTime platform_get_utc_ms_stm32(void) {
    uint32_t ms = (uint32_t)(__HAL_TIM_GET_COUNTER(&htim2) / 1000);  // TIM2以1MHz计数
    return (PlatformTime)g_utc_seconds * 1000 + ms;
}

void platform_delay_ms_stm32(uint32 ms) {
    if (ms == 0) return;
    vTaskDelay(pdMS_TO_TICKS(ms));
}

// TIM2中断服务函数（1ms触发一次，用于UTC时间计数）
void TIM2_IRQHandler(void) {
    if (__HAL_TIM_GET_FLAG(&htim2, TIM_FLAG_UPDATE) != RESET) {
        __HAL_TIM_CLEAR_FLAG(&htim2, TIM_FLAG_UPDATE);
        g_utc_seconds++;  // 每1秒递增一次
    }
}

// -------------------------- 定时器实现（补充启停控制） --------------------------
// FreeRTOS定时器回调包装器
static void timer_callback_wrapper(TimerHandle_t timer) {
    Stm32Timer* stm_timer = (Stm32Timer*)pvTimerGetTimerID(timer);
    if (stm_timer && stm_timer->func && stm_timer->is_running) {
        stm_timer->func(stm_timer->arg);
    }
}

PlatformTimer platform_timer_create_stm32(PlatformTimerFunc func, void* arg, uint32 period_ms) {
    if (func == NULL || period_ms == 0) return NULL;

    // 分配定时器结构
    Stm32Timer* timer = (Stm32Timer*)pvPortMalloc(sizeof(Stm32Timer));
    if (!timer) {
        agent_log_error("Failed to allocate timer");
        return NULL;
    }

    // 创建FreeRTOS软件定时器
    timer->timer = xTimerCreate(
        "STM32_Timer",
        pdMS_TO_TICKS(period_ms),
        pdTRUE,  // 周期性
        timer,   // 传递自身作为ID
        timer_callback_wrapper
    );
    if (!timer->timer) {
        agent_log_error("xTimerCreate failed");
        vPortFree(timer);
        return NULL;
    }

    timer->func = func;
    timer->arg = arg;
    timer->period_ms = period_ms;
    timer->is_running = BOOL8_FALSE;  // 初始停止状态

    return (PlatformTimer)timer;
}

bool8 platform_timer_start_stm32(PlatformTimer timer) {
    if (!timer) return BOOL8_FALSE;
    Stm32Timer* stm_timer = (Stm32Timer*)timer;

    if (xTimerStart(stm_timer->timer, 0) == pdPASS) {
        stm_timer->is_running = BOOL8_TRUE;
        return BOOL8_TRUE;
    }
    return BOOL8_FALSE;
}

bool8 platform_timer_stop_stm32(PlatformTimer timer) {
    if (!timer) return BOOL8_FALSE;
    Stm32Timer* stm_timer = (Stm32Timer*)timer;

    if (xTimerStop(stm_timer->timer, 0) == pdPASS) {
        stm_timer->is_running = BOOL8_FALSE;
        return BOOL8_TRUE;
    }
    return BOOL8_FALSE;
}

void platform_timer_destroy_stm32(PlatformTimer timer) {
    if (timer != NULL) {
        Stm32Timer* stm_timer = (Stm32Timer*)timer;
        xTimerDelete(stm_timer->timer, 0);
        vPortFree(stm_timer);
    }
}

// -------------------------- 内存接口实现（补充基础分配） --------------------------
void* platform_malloc_stm32(uint32 size) {
    return pvPortMalloc(size);  // FreeRTOS堆分配
}

void* platform_calloc_stm32(uint32 num, uint32 size) {
    void* ptr = pvPortMalloc(num * size);
    if (ptr) memset(ptr, 0, num * size);
    return ptr;
}

void platform_free_stm32(void* ptr) {
    if (ptr) vPortFree(ptr);
}

// 对齐内存分配实现
typedef struct {
    void* original_ptr;  // 原始分配地址
} AlignMemHeader;

void* platform_malloc_aligned_stm32(uint32 size, uint32 align) {
    if (size == 0 || align == 0 || (align & (align - 1)) != 0) {
        agent_log_error("Invalid alignment");
        return NULL;  // 对齐值必须是2的幂
    }

    // 计算总分配大小（包含头部和对齐填充）
    uint32 total_size = size + align + sizeof(AlignMemHeader);
    AlignMemHeader* header = (AlignMemHeader*)pvPortMalloc(total_size);
    if (header == NULL) {
        agent_log_error("Aligned malloc failed");
        return NULL;
    }

    header->original_ptr = header;
    uintptr_t align_addr = (uintptr_t)(header + 1);
    // 向上对齐到指定边界
    align_addr = (align_addr + align - 1) & ~(align - 1);

    return (void*)align_addr;
}

void platform_free_aligned_stm32(void* ptr) {
    if (ptr == NULL) return;

    // 找回原始分配地址
    AlignMemHeader* header = (AlignMemHeader*)((uintptr_t)ptr - sizeof(AlignMemHeader));
    vPortFree(header->original_ptr);
}

// -------------------------- 随机数实现（补充加密级） --------------------------
bool8 platform_rand_generate_stm32(uint8* buf, uint32 len) {
    if (buf == NULL || len == 0) return BOOL8_FALSE;

    // 普通随机数（使用硬件RNG但不严格校验）
    for (uint32 i = 0; i < len; ) {
        uint32 rand_val;
        if (HAL_RNG_GenerateRandomNumber(&hrng, &rand_val) != HAL_OK) {
            agent_log_error("RNG generate failed");
            return BOOL8_FALSE;
        }
        // 按字节填充
        for (uint32 j = 0; j < 4 && i < len; j++) {
            buf[i++] = (uint8)(rand_val >> (j * 8));
        }
    }
    return BOOL8_TRUE;
}

bool8 platform_rand_generate_crypto_stm32(uint8* buf, uint32 len) {
    if (buf == NULL || len == 0) return BOOL8_FALSE;

    // 加密级随机数（严格校验硬件RNG状态）
    if (HAL_RNG_GetState(&hrng) != HAL_RNG_STATE_READY) {
        agent_log_error("RNG not ready");
        return BOOL8_FALSE;
    }

    for (uint32 i = 0; i < len; ) {
        uint32 rand_val;
        HAL_StatusTypeDef status = HAL_RNG_GenerateRandomNumber(&hrng, &rand_val);
        if (status != HAL_OK) {
            agent_log_error("Crypto RNG failed: %d", status);
            return BOOL8_FALSE;
        }
        // 按字节填充
        for (uint32 j = 0; j < 4 && i < len; j++) {
            buf[i++] = (uint8)(rand_val >> (j * 8));
        }
    }
    return BOOL8_TRUE;
}

// -------------------------- 文件系统接口实现（适配SPI Flash等） --------------------------
bool8 platform_file_exists_stm32(const char* path) {
    // STM32通常使用SPI Flash或SD卡，此处简化实现
    // 实际项目需替换为具体文件系统检查（如FatFs的f_stat）
    (void)path;
    return BOOL8_FALSE;  // 默认返回不存在
}

uint32 platform_get_file_size_stm32(const char* path) {
    // 实际项目需替换为具体文件系统实现
    (void)path;
    return 0;
}

// -------------------------- 字符串/内存操作实现 --------------------------
uint32 platform_memset_s_stm32(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len) {
    if (!dest || dest_max_len == 0) return 0;
    uint32 actual_len = (set_len < dest_max_len) ? set_len : dest_max_len;
    memset(dest, value, actual_len);
    return actual_len;
}

int32 platform_strcmp_stm32(const char* str1, const char* str2) {
    return strcmp(str1, str2);
}

char* platform_strchr_stm32(const char* str, int32 ch) {
    return strchr(str, ch);
}

// -------------------------- 日志接口实现（输出到调试串口） --------------------------
void platform_log_output_stm32(const char* module, const char* fmt, va_list args) {
    char log_buf[256] = {0};
    vsnprintf(log_buf, sizeof(log_buf), fmt, args);
    agent_log_info(module, "%s", log_buf);
}

// -------------------------- 运行时文件名接口 --------------------------
uint32 platform_get_runtime_filename_stm32(char* buf, uint32 buf_len) {
    if (buf == NULL || buf_len == 0) return 0;

    const char* firmware_name = "SNMP_AGENT_STM32.bin";
    uint32 copy_len = strlen(firmware_name);
    if (copy_len >= buf_len) copy_len = buf_len - 1;
    strncpy(buf, firmware_name, copy_len);
    buf[copy_len] = '\0';

    return copy_len;
}

// -------------------------- 平台反初始化 --------------------------
void platform_deinit_stm32(void) {
    HAL_TIM_Base_Stop_IT(&htim2);
    HAL_TIM_Base_Stop(&htim7);
    HAL_RNG_DeInit(&hrng);
    
    __HAL_RCC_TIM2_CLK_DISABLE();
    __HAL_RCC_TIM7_CLK_DISABLE();
    __HAL_RCC_RNG_CLK_DISABLE();
}

// -------------------------- 硬件初始化函数 --------------------------
void MX_TIM7_Init(void) {
    __HAL_RCC_TIM7_CLK_ENABLE();

    htim7.Instance = TIM7;
    htim7.Init.Prescaler = 83;  // 84MHz / 84 = 1MHz（1us计数）
    htim7.Init.CounterMode = TIM_COUNTERMODE_UP;
    htim7.Init.Period = 0xFFFF;  // 最大计数65535us
    htim7.Init.AutoReloadPreload = TIM_AUTORELOAD_PRELOAD_DISABLE;
    if (HAL_TIM_Base_Init(&htim7) != HAL_OK) {
        Error_Handler();
    }
    HAL_TIM_Base_Start(&htim7);
}

void MX_TIM2_Init(void) {
    __HAL_RCC_TIM2_CLK_ENABLE();

    htim2.Instance = TIM2;
    htim2.Init.Prescaler = 83;  // 84MHz / 84 = 1MHz（1us计数）
    htim2.Init.CounterMode = TIM_COUNTERMODE_UP;
    htim2.Init.Period = 999999;  // 1秒触发一次更新中断
    htim2.Init.AutoReloadPreload = TIM_AUTORELOAD_PRELOAD_DISABLE;
    if (HAL_TIM_Base_Init(&htim2) != HAL_OK) {
        Error_Handler();
    }

    // 使能更新中断
    HAL_NVIC_SetPriority(TIM2_IRQn, 5, 0);
    HAL_NVIC_EnableIRQ(TIM2_IRQn);
    HAL_TIM_Base_Start_IT(&htim2);
}

void MX_RNG_Init(void) {
    __HAL_RCC_RNG_CLK_ENABLE();

    hrng.Instance = RNG;
    if (HAL_RNG_Init(&hrng) != HAL_OK) {
        Error_Handler();
    }
}

// -------------------------- 错误处理 --------------------------
void Error_Handler(void) {
    while (1) {
        HAL_Delay(500);
        // 实际项目中可添加LED闪烁代码指示错误
    }
}

#ifdef USE_FULL_ASSERT
void assert_failed(uint8 *file, uint32 line) {
    while (1) {
        HAL_Delay(100);
        // 实际项目中可添加LED快速闪烁代码指示断言失败
    }
}
#endif
#endif
