#!/bin/bash
# snmp_code/code/nms_sdk/build_all.sh  # 芯祥联定制：适配SDK目录改名nms_sdk
# 芯祥联科技 - SNMP协议栈多平台编译脚本
# 最终版：添加平台宏定义(WIN32/LINUX/STM32) + 修复所有编译错误 + 开源os目录
# 支持平台：Linux/Windows(MinGW)/STM32(ARM Cortex-M4)
# 优化点：1. Windows平台补全头文件路径 2. STM32链接脚本前置检查 3. 增加权限提示 4. 容错性增强 5. 强化临时文件清理

set -euo pipefail

# ==================== 【核心配置 - 芯祥联定制】 ====================
OPEN_SRC_DIRS=(
    "./network"       
    "./examples"      
    "./comm"          
    "./os"            
)
# 芯祥联定制：SDK库根目录适配nms_sdk目录改名（原./lib → 适配上级nms_sdk/lib）
SDK_LIB_ROOT="../nms_sdk/lib"  
INC_DIRS=(
    "./comm"              
    "./os"                
    "./"                  
    "./network"           
    "./examples"          
    "../nms_sdk/src/include"  # 芯祥联定制：补充nms_sdk头文件路径
)
OUTPUT_ROOT="./bin"
# 芯祥联定制：产物名添加公司标识，便于区分
TARGET_BASE_NAME="xxl_nms_snmp_demo"  
STM32_CPU="cortex-m4"
STM32_LINK_SCRIPT="./stm32/linker.ld"
STM32_LD_FLAGS="-nostdlib -lc -lm -lnosys -Wl,--gc-sections"

# ==================== 【全局变量】 ====================
OS_TYPE=""
CC=""
SDK_LIB_FILE=""
TARGET_NAME=""
BUILD_PLATFORM="all"
ERROR_LOG="./compile_error.log"
SCRIPT_DIR=$(cd $(dirname $0); pwd)  # 脚本所在绝对路径

# ==================== 【工具函数】 ====================
show_help() {
    echo -e "\033[32m==== 芯祥联科技 - NMS SNMP协议栈多平台编译脚本 ====\033[0m"  # 芯祥联定制：添加公司名称
    echo "核心特性："
    echo "  1. 添加平台宏定义：WIN32/ LINUX/ STM32（触发代码对应分支）"
    echo "  2. 修复 malloc/calloc/free/printf 等未声明问题"
    echo "  3. 提示函数参数不匹配问题，支持 sdk/os 开源修改"
    echo "  4. 自动扫描 OPEN_SRC_DIRS 下所有 .c 文件（含examples目录的snmp_api_demo.c）"
    echo "  5. 强化临时文件清理：编译完成后彻底删除所有.o文件（芯祥联定制）"
    echo "用法：$0 [-p|--platform <平台>] [-h|--help]"
    echo "支持平台：linux/windows/stm32/all"
    echo "示例："
    echo "  ./build_all.sh -p windows    # 仅编译Windows平台"
    echo "  ./build_all.sh -p all        # 编译所有平台（默认）"
    exit 0
}

# 检查依赖工具
check_deps() {
    local platform="$1"
    case "${platform}" in
        linux)
            if ! command -v gcc &>/dev/null; then
                echo -e "\033[31m[ERROR] Linux平台缺少gcc！请安装：sudo apt install gcc\033[0m"
                exit 1
            fi
            ;;
        windows)
            if ! command -v gcc &>/dev/null && [ ! -f "D:/Program/msys2/mingw64/bin/gcc.exe" ]; then
                echo -e "\033[31m[ERROR] Windows平台未找到MinGW gcc！请安装MSYS2并配置环境变量\033[0m"
                exit 1
            fi
            ;;
        stm32)
            if ! command -v arm-none-eabi-gcc &>/dev/null; then
                echo -e "\033[31m[ERROR] STM32平台缺少ARM交叉编译器！请安装：arm-none-eabi-gcc\033[0m"
                exit 1
            fi
            if [ ! -f "${STM32_LINK_SCRIPT}" ]; then
                echo -e "\033[31m[ERROR] STM32链接脚本不存在：${STM32_LINK_SCRIPT}\033[0m"
                echo -e "\033[33m[提示] 请将STM32链接脚本放入指定路径，或修改脚本中 STM32_LINK_SCRIPT 变量\033[0m"
                exit 1
            fi
            ;;
    esac
}

parse_args() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
            -p|--platform) 
                BUILD_PLATFORM="$2"
                shift 2 
                ;;
            -h|--help) 
                show_help 
                ;;
            *) 
                echo -e "\033[31m[ERROR] 无效参数：$1\033[0m"
                show_help 
                ;;
        esac
    done
    if [[ ! "${BUILD_PLATFORM}" =~ ^(linux|windows|stm32|all)$ ]]; then
        echo -e "\033[31m[ERROR] 无效平台！仅支持：linux/windows/stm32/all\033[0m"
        exit 1
    fi
}

init_platform() {
    local platform="$1"
    rm -f "${ERROR_LOG}"

    case "${platform}" in
        linux)
            OS_TYPE="Linux"
            CC="gcc"
            # 芯祥联定制：适配nms_sdk目录下的库文件路径
            SDK_LIB_FILE="${SDK_LIB_ROOT}/linux/lib_nms_sdk_linux.a"
            TARGET_NAME="${TARGET_BASE_NAME}_${platform}"
            OUTPUT_DIR="${OUTPUT_ROOT}/linux"
            # 核心：添加 -DLINUX 宏定义，触发代码中 LINUX 分支
            CFLAGS="-O2 -std=c99 -DLINUX -Wall -Wno-error -fPIC -include stdio.h -include stdlib.h"
            LDFLAGS=""
            ;;
        windows)
            OS_TYPE="Windows"
            if command -v gcc &>/dev/null; then
                CC="gcc"
                echo -e "\033[32m[INFO] 使用MinGW路径：$(which gcc)\033[0m"
            else
                CC="D:/Program/msys2/mingw64/bin/gcc.exe"
            fi
            # 芯祥联定制：适配nms_sdk目录下的库文件路径
            SDK_LIB_FILE="${SDK_LIB_ROOT}/windows/lib_nms_sdk_windows.lib"
            TARGET_NAME="${TARGET_BASE_NAME}_${platform}.exe"
            OUTPUT_DIR="${OUTPUT_ROOT}/windows"
            # 修复：Windows平台也添加完整宏定义+头文件包含
            CFLAGS="-O2 -std=c99 -DWIN32 -D_WIN32 -D__USE_MINGW_ANSI_STDIO=1 -D__STDC_WANT_LIB_EXT1__=1 -Wall -Wno-error -Wno-builtin-declaration-mismatch -Wno-unknown-pragmas -Wno-incompatible-pointer-types -include stdio.h -include stdlib.h"
            LDFLAGS="-lws2_32 -lbcrypt -lkernel32 -mconsole"
            ;;
        stm32)
            OS_TYPE="STM32"
            CC="arm-none-eabi-gcc"
            # 芯祥联定制：适配nms_sdk目录下的库文件路径
            SDK_LIB_FILE="${SDK_LIB_ROOT}/stm32/lib_nms_sdk_stm32.a"
            TARGET_NAME="${TARGET_BASE_NAME}_${platform}.elf"
            OUTPUT_DIR="${OUTPUT_ROOT}/stm32"
            # 核心：添加 -DSTM32 宏定义，触发代码中 STM32 分支
            CFLAGS="-O2 -std=c99 -DSTM32 -mcpu=${STM32_CPU} -mthumb -ffunction-sections -fdata-sections -Wall -Wno-error -include stdio.h -include stdlib.h"
            LDFLAGS="-mcpu=${STM32_CPU} -mthumb -T${STM32_LINK_SCRIPT} ${STM32_LD_FLAGS}"
            ;;
    esac

    # 创建输出目录（容错）
    mkdir -p "${OUTPUT_DIR}" || {
        echo -e "\033[31m[ERROR] 创建${platform}输出目录失败！\033[0m"
        exit 1
    }

    # 检查SDK库文件（芯祥联定制：优化提示文案，匹配nms_sdk目录）
    if [[ ! -f "${SDK_LIB_FILE}" ]]; then
        echo -e "\033[31m[ERROR] ${platform} SDK库不存在！请先编译芯祥联SNMP闭源库，放入：${SDK_LIB_FILE}\033[0m"
        echo -e "\033[33m[提示] SDK库文件要求：\033[0m"
        echo -e "\033[33m  - Linux: ${SDK_LIB_ROOT}/linux/lib_nms_sdk_linux.a\033[0m"
        echo -e "\033[33m  - Windows: ${SDK_LIB_ROOT}/windows/lib_nms_sdk_windows.lib\033[0m"
        echo -e "\033[33m  - STM32: ${SDK_LIB_ROOT}/stm32/lib_nms_sdk_stm32.a\033[0m"
        exit 1
    fi

    # 生成头文件参数（所有平台统一）
    INC_PARAMS=""
    for dir in "${INC_DIRS[@]}"; do
        local abs_dir="${SCRIPT_DIR}/${dir}"
        if [[ -d "${abs_dir}" ]]; then
            INC_PARAMS+=" -I${abs_dir}"
        fi
    done

    # 全局导出变量
    export OS_TYPE CC SDK_LIB_FILE TARGET_NAME OUTPUT_DIR CFLAGS LDFLAGS INC_PARAMS

    echo -e "\033[32m[INFO] ✅ ${platform} 配置初始化完成（芯祥联科技）\033[0m"  # 芯祥联定制：添加公司标识
    echo -e "\033[32m[INFO] 已添加平台宏定义：${platform^^}（代码中 #ifdef ${platform^^} 分支会触发）\033[0m"
    echo -e "\033[32m[INFO] SDK库路径：${SDK_LIB_FILE}\033[0m"
    echo -e "\033[32m[INFO] 产物输出路径：${OUTPUT_DIR}/${TARGET_NAME}\033[0m"
    echo -e "\033[32m[INFO] 头文件搜索路径：${INC_PARAMS}\033[0m"
}

# 收集所有合法的.c文件
collect_c_files() {
    local c_files=()
    for dir in "${OPEN_SRC_DIRS[@]}"; do
        local abs_dir="${SCRIPT_DIR}/${dir}"
        if [[ ! -d "${abs_dir}" ]]; then
            echo -e "\033[33m[WARN] SDK目录不存在，跳过：${abs_dir}\033[0m" >&2
            continue
        fi
        # 递归搜索.c文件（排除隐藏文件）
        while IFS= read -r -d '' file; do
            c_files+=("${file}")
        done < <(find "${abs_dir}" -name "*.c" -type f -not -name ".*" -print0)
    done

    # 过滤有效文件
    local valid_c_files=()
    for file in "${c_files[@]}"; do
        if [[ -f "${file}" && "${file}" =~ \.c$ ]]; then
            valid_c_files+=("${file}")
        fi
    done

    if [[ ${#valid_c_files[@]} -eq 0 ]]; then
        echo -e "\033[31m[ERROR] 未找到任何合法的.c文件！\033[0m"
        echo -e "\033[33m[提示] 请检查 OPEN_SRC_DIRS 配置的目录是否存在.c文件（如examples/snmp_api_demo.c）\033[0m"
        exit 1
    fi

    printf "%s\n" "${valid_c_files[@]}"
}

# 编译单个平台（芯祥联定制：强化临时文件清理）
build_platform() {
    local platform="$1"
    echo -e "\033[32m\n====================================\033[0m"
    echo -e "\033[32m[INFO] 芯祥联科技 - 开始编译${platform}平台...\033[0m"  # 芯祥联定制：添加公司标识
    echo -e "\033[32m====================================\033[0m"

    # 前置检查
    check_deps "${platform}"
    init_platform "${platform}"

    # 收集C文件
    local c_files=($(collect_c_files))
    local c_file_count=${#c_files[@]}
    echo -e "\033[32m[INFO] 找到${c_file_count}个SDK内合法.c文件（含examples/snmp_api_demo.c）\033[0m"

    # 编译.c文件为.o文件
    local obj_files=""
    echo -e "\033[32m[INFO] 开始编译.c文件...\033[0m"
    for c_file in "${c_files[@]}"; do
        local obj_basename=$(basename "${c_file}" .c)
        local obj_file="${OUTPUT_DIR}/${obj_basename}.o"
        
        echo -e "\033[32m[INFO] 编译：${c_file} → ${obj_file}\033[0m"
        compile_cmd="${CC} ${CFLAGS} ${INC_PARAMS} -c ${c_file} -o ${obj_file}"
        
        # 调试模式：打印编译命令
        # echo -e "\033[32m[DEBUG] 编译命令：${compile_cmd}\033[0m"
        
        if ! ${compile_cmd} 2>"${ERROR_LOG}"; then
            echo -e "\033[31m[ERROR] 编译失败：${c_file}\033[0m"
            echo -e "\033[31m[错误详情]：$(cat ${ERROR_LOG})\033[0m"
            
            # 针对性修复提示（芯祥联定制：匹配自研SNMP协议栈代码）
            if [[ "${c_file}" == *"platform.c"* ]]; then
                echo -e "\033[31m[修复建议]：\033[0m"
                echo -e "\033[31m  1. 缺少头文件：在 ${c_file} 开头添加 #include <stdlib.h>\033[0m"
                echo -e "\033[31m  2. 函数参数不匹配：platform_malloc_aligned 声明（platform.h）有3个参数，定义（platform.c）只有2个，请对齐\033[0m"
                echo -e "\033[31m     示例：void* platform_malloc_aligned(uint32 size, uint32 align, PlatformMemType type) {\033[0m"
            elif [[ "${c_file}" == *"agent_log.c"* ]]; then
                echo -e "\033[31m[修复建议]：在 ${c_file} 开头添加 #include <stdio.h> 和 #include <string.h>\033[0m"
            elif [[ "${c_file}" == *"snmp_api_demo.c"* ]]; then
                echo -e "\033[31m[修复建议]：检查芯祥联nms_snmp_api.h/platform.h是否存在，且路径正确\033[0m"  # 芯祥联定制
            fi
            
            rm -f "${ERROR_LOG}"
            exit 1
        fi
        obj_files+=" ${obj_file}"
    done
    rm -f "${ERROR_LOG}"

    # 链接最终产物
    local target_path="${OUTPUT_DIR}/${TARGET_NAME}"
    echo -e "\033[32m[INFO] 开始链接最终产物：${target_path}\033[0m"
    link_cmd="${CC} ${obj_files} ${SDK_LIB_FILE} ${LDFLAGS} -o ${target_path}"
    
    # 调试模式：打印链接命令
    # echo -e "\033[32m[DEBUG] 链接命令：${link_cmd}\033[0m"
    
    if ! ${link_cmd} 2>"${ERROR_LOG}"; then
        echo -e "\033[31m[ERROR] ${platform}平台链接失败！\033[0m"
        echo -e "\033[31m[错误详情]：$(cat ${ERROR_LOG})\033[0m"
        rm -f "${ERROR_LOG}"
        exit 1
    fi

    # 芯祥联定制：强化临时文件清理（彻底删除所有.o文件，防止漏删）
    echo -e "\033[32m[INFO] 清理临时.o文件...\033[0m"
    if [[ "${platform}" == "windows" ]]; then
        eval rm -f ${obj_files}  # 处理Windows路径空格
    else
        rm -f ${obj_files}
    fi
    rm -f "${OUTPUT_DIR}"/*.o  # 兜底删除输出目录下所有.o文件
    rm -f "${ERROR_LOG}"       # 清理错误日志
    echo -e "\033[32m[INFO] ✅ 临时文件已全部清理完成\033[0m"

    # 编译完成提示（芯祥联定制：添加公司业务提示）
    echo -e "\033[32m[INFO] ✅ 芯祥联科技 - ${platform}平台编译完成！\033[0m"
    echo -e "\033[32m[INFO] 最终产物：${target_path}\033[0m"
    echo -e "\033[33m[提示] 执行TRAP监听功能时，需以管理员/root权限运行（162端口为特权端口）\033[0m"
    echo -e "\033[33m[芯祥联提示] 该产物基于自研SNMP协议栈编译，支持航空/工业/通信领域设备监控\033[0m"  # 芯祥联定制
}

# 编译所有平台
build_all() {
    build_platform "linux"
    build_platform "windows"
    build_platform "stm32"
}

# 主逻辑
main() {
    parse_args "$@"
    
    echo -e "\033[32m==== 芯祥联科技 - NMS SNMP SDK 多平台编译启动 ====\033[0m"  # 芯祥联定制
    echo -e "\033[32m[INFO] 脚本工作目录：${SCRIPT_DIR}\033[0m"
    echo -e "\033[32m[INFO] 目标编译平台：${BUILD_PLATFORM}\033[0m"
    echo -e "\033[32m[INFO] 芯祥联科技 - 自研SNMP协议栈适配：Linux/Windows/STM32\033[0m"  # 芯祥联定制

    case "${BUILD_PLATFORM}" in
        linux) build_platform "linux" ;;
        windows) build_platform "windows" ;;
        stm32) build_platform "stm32" ;;
        all) build_all ;;
    esac

    echo -e "\033[32m\n====================================\033[0m"
    echo -e "\033[32m[INFO] ✅ 芯祥联科技 - 所有指定平台编译完成！\033[0m"  # 芯祥联定制
    echo -e "\033[32m[INFO] 产物总目录：${OUTPUT_ROOT}\033[0m"
    echo -e "\033[32m[INFO] 各平台宏定义已生效：WIN32(Windows)/LINUX(Linux)/STM32(STM32)\033[0m"
    echo -e "\033[32m[INFO] 芯祥联科技 - 产物适配航空/工业/通信领域SNMP设备监控场景\033[0m"  # 芯祥联定制
    echo -e "\033[32m====================================\033[0m"
}

# 启动主逻辑
main "$@"
