﻿/**
 * File Name: snmp_api_demo.c
 * Description: 单窗口SNMP监听+命令发送（后台监听+前台输入）
 * Author: ZXY
 * Platform: Linux/Windows(MinGW)
 * Note: Windows需管理员权限运行（162端口）
 */
#include "nms_snmp_api.h"
#include "platform.h"
#include "type.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// ==================== 平台适配宏 ====================
#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#define SET_CONSOLE_UTF8() do { \
    SetConsoleOutputCP(CP_UTF8); \
    SetConsoleCP(CP_UTF8); \
} while(0)
#define PLATFORM_CLEANUP() WSACleanup()
#define PLATFORM_INIT() do { \
    WSADATA wsaData; \
    WSAStartup(MAKEWORD(2, 2), &wsaData); \
    SET_CONSOLE_UTF8(); \
    setvbuf(stdout, NULL, _IONBF, 0); /* 禁用输出缓冲 */ \
} while(0)
// Windows线程定义
#define THREAD_HANDLE HANDLE
#define THREAD_FUNC DWORD WINAPI
#define CREATE_THREAD(func, arg) CreateThread(NULL, 0, func, arg, 0, NULL)
#define WAIT_THREAD(hdl) WaitForSingleObject(hdl, INFINITE)
#define CLOSE_THREAD(hdl) CloseHandle(hdl)

#elif defined(LINUX)
#include <signal.h>
#include <unistd.h>
#include <pthread.h>
#define PLATFORM_CLEANUP()
#define PLATFORM_INIT() do { \
    setvbuf(stdout, NULL, _IONBF, 0); /* 禁用输出缓冲 */ \
} while(0)
// Linux线程定义
#define THREAD_HANDLE pthread_t
#define THREAD_FUNC void*
#define CREATE_THREAD(func, arg) pthread_create(&func, NULL, arg, NULL)
#define WAIT_THREAD(hdl) pthread_join(hdl, NULL)
#define CLOSE_THREAD(hdl) ((void)0)
#endif

// ==================== 全局配置 ====================
#define AGENT_IP          "127.0.0.1"
#define AGENT_PORT        161
#define TRAP_LISTEN_PORT  162
#define COMMUNITY_PUBLIC  "public"
#define COMMUNITY_PRIVATE "private"
#define TIMEOUT_MS        5000

// 核心OID
#define OID_SYS_DESCR     "1.3.6.1.2.1.1.1.0"  // 系统描述
#define OID_SYS_UPTIME    "1.3.6.1.2.1.1.3.0"  // 系统运行时间
#define OID_SYS_CONTACT   "1.3.6.1.2.1.1.4.0"  // 联系邮箱
#define OID_LED_CONTROL   "1.3.6.1.4.1.12345.1.2" // LED控制（触发Trap/Inform）
#define OID_MIB_TREE      "1.3.6.1.2.1.1"      // 系统组MIB树

// ==================== 全局控制标记（核心修改） ====================
static volatile bool8 g_listen_running = BOOL8_FALSE; // 监听线程运行标记
static volatile bool8 g_exit_flag = BOOL8_FALSE;      // 程序退出标记
static volatile bool8 g_snmp_inited = BOOL8_FALSE;    // SNMP初始化标记
static THREAD_HANDLE g_listen_thread;                 // 监听线程句柄
static volatile bool8 g_timer_running = BOOL8_FALSE;  // 定时器运行标记
static THREAD_HANDLE g_timer_thread;                  // 定时器线程句柄

// ==================== 信号处理（退出） ====================
#ifdef WIN32
BOOL WINAPI console_handler(DWORD dwCtrlType) {
    if (dwCtrlType == CTRL_C_EVENT) {
        g_exit_flag = BOOL8_TRUE;
        g_listen_running = BOOL8_FALSE;
        printf("\n[INFO] 接收到退出信号，正在清理资源...\n");
        return TRUE;
    }
    return FALSE;
}
#elif defined(LINUX)
void sigint_handler(int sig) {
    (void)sig;
    g_exit_flag = BOOL8_TRUE;
    g_listen_running = BOOL8_FALSE;
    printf("\n[INFO] 接收到退出信号，正在清理资源...\n");
}
#endif
static void platform_delay_ms(uint32 ms) {
    Sleep(ms);
}

// 1ms定时器线程（极简实现）
#ifdef WIN32
THREAD_FUNC timer_thread_proc(LPVOID lpParam) {
#else
THREAD_FUNC timer_thread_proc(void* arg) {
#endif
    (void)lpParam;
    g_timer_running = BOOL8_TRUE;
    while (g_timer_running && !g_exit_flag) {
        api_snmp_1ms_timer_tick();  // 1ms调用一次核心接口
        platform_delay_ms(1);       // 跨平台1ms延时
    }
    return 0;
}

// ==================== 单例SNMP初始化（最小修改） ====================
static void safe_snmp_init(void) {
    if (!g_snmp_inited) {
        api_snmp_init();
        g_snmp_inited = BOOL8_TRUE;
        // Windows端口复用（解决162端口独占）
#ifdef WIN32
        int reuse = 1;
        SOCKET sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char*)&reuse, sizeof(reuse));
        closesocket(sock);
#endif
        // ★ 新增：SNMP初始化完成后，启动1ms定时器线程
        g_timer_thread = CREATE_THREAD(timer_thread_proc, NULL);
        g_timer_running = (g_timer_thread != NULL); // 标记启动状态
    }
}

// ==================== 通用响应回调函数（保留原有） ====================
void response_callback(char* agent_ip, uint16 agent_port, char* username,
                      ApiSnmpCmdType cmd_type, ApiSnmpResponse* response, void* user_data) {
    if (!response) {
        printf("响应数据无效\n");
        return;
    }

    const char* cmd_name = "GET";
    if (cmd_type == API_SNMP_CMD_GET_NEXT) cmd_name = "GET-NEXT";
    else if (cmd_type == API_SNMP_CMD_GET_BULK) cmd_name = "GET-BULK";

    printf("\n=== %s 响应结果（%s:%d）===\n", cmd_name, agent_ip, agent_port);
    if (response->error != API_SNMP_ERR_SUCCESS) {
        printf("执行失败：API错误=%s（错误码：%d），协议错误状态=%d\n",
               api_snmp_error_to_str(response->error), response->error, response->snmp_error_status);
        return;
    }

    if (response->varbind_count == 0) {
        printf("无有效变量绑定数据\n");
        return;
    }

    for (uint8 i = 0; i < response->varbind_count; i++) {
        ApiSnmpVarBind* vb = &response->varbinds[i];
        printf("  [%d] OID：%s = ", i + 1, vb->oid);
        if (vb->type == API_SNMP_BER_TIMETICKS) {
            printf("TIMETICKS: %u (%.2f秒)\n", vb->value.uint_val, vb->value.uint_val / 100.0);
        } else if (vb->type == API_SNMP_BER_OCTET_STRING) {
            printf("STRING: %s\n", vb->value.str_val);
        } else if (vb->type == API_SNMP_BER_INTEGER) {
            printf("INTEGER: %d\n", vb->value.int_val);
        } else if (vb->type == API_SNMP_BER_UINTEGER32) {
            printf("UINT32: %u\n", vb->value.uint_val);
        } else {
            printf("TYPE(0x%02X)：%d字节\n", vb->type, vb->value_len);
        }
    }
    printf("========================================\n");
}

// ==================== SET操作响应回调（保留原有） ====================
void set_response_callback(char* agent_ip, uint16 agent_port, char* username,
                          ApiSnmpCmdType cmd_type, ApiSnmpResponse* response, void* user_data) {
    if (!response) {
        printf("SET响应无效\n");
        return;
    }

    printf("\n=== SET 操作结果（%s:%d）===\n", agent_ip, agent_port);
    if (response->error == API_SNMP_ERR_SUCCESS && response->snmp_error_status == 0) {
        const char* oid = (const char*)user_data;
        int led_status = atoi(strrchr(oid, '=') ? strrchr(oid, '=') + 1 : "0");
        // 【修改1】替换✅为[OK]
        printf(" [OK] SET操作成功！OID：%s = %s\n", OID_LED_CONTROL,
               led_status == 1 ? "开启（触发Inform）" : "关闭（触发Trap）");
    } else {
        printf(" SET操作失败：API错误=%s，协议错误状态=%d\n",
               api_snmp_error_to_str(response->error), response->snmp_error_status);
    }
    printf("========================================\n");
}

// ==================== TRAP/INFORM接收回调（核心修改：恢复提示符） ====================
void trap_callback(const char* agent_ip, uint16 agent_port,
                   const ApiSnmpTrapMessage* trap_msg, ApiSnmpError error, void* user_data) {
    if (error != API_SNMP_ERR_SUCCESS) {
        // 【修改2】替换📢为[INFO]
        printf("\n [INFO] TRAP/INFORM接收失败：%s（错误码：%d）\n", api_snmp_error_to_str(error), error);
        printf("> "); fflush(stdout); // 恢复提示符
        return;
    }

    // 打印TRAP消息
    printf("\n========================================\n");
    // 【修改3】替换📢为[INFO]
    printf("[INFO] 收到TRAP/INFORM 来源：%s:%d（版本：%s）\n",
           agent_ip, agent_port, trap_msg->version == API_SNMP_VERSION_1 ? "v1" : "v2c");
    printf("安全信息：共同体名 = %s\n", trap_msg->sec.community);
    printf("变量数据（共%d个）：\n", trap_msg->varbind_count);
    
    for (uint8 i = 0; i < trap_msg->varbind_count; i++) {
        const ApiSnmpVarBind* vb = &trap_msg->varbinds[i];
        printf("  [%d] OID：%s = ", i + 1, vb->oid);
        switch (vb->type) {
            case API_SNMP_BER_OCTET_STRING: printf("STRING: %s\n", vb->value.str_val); break;
            case API_SNMP_BER_TIMETICKS: printf("TIMETICKS: %u (%.2f秒)\n", vb->value.uint_val, vb->value.uint_val / 100.0); break;
            case API_SNMP_BER_OID: printf("OID: %s\n", vb->value.str_val); break;
            default: printf("TYPE(0x%02X)：%d字节\n", vb->type, vb->value_len); break;
        }
    }
    printf("----------------------------------------\n");
    printf("> "); fflush(stdout); // 核心修改：恢复输入提示符
}

// ==================== 原有功能函数（仅修改初始化） ====================
// SNMPv2c GET
void snmp_v2c_get(void) {
    ApiSnmpError err;
    safe_snmp_init(); // 替换原有api_snmp_init()
    api_snmp_register_response_callback(response_callback);

    ApiSnmpAgentConfig agent_cfg = {.version=API_SNMP_VERSION_2C, .community=COMMUNITY_PUBLIC};
    err = api_snmp_agent_create(AGENT_IP, AGENT_PORT, &agent_cfg);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("创建Agent失败：%s\n", api_snmp_error_to_str(err));
        return;
    }

    ApiSnmpVarBind varbinds[3];
    err = api_snmp_create_get_varbind(OID_SYS_DESCR, &varbinds[0]);
    err |= api_snmp_create_get_varbind(OID_SYS_UPTIME, &varbinds[1]);
    err |= api_snmp_create_get_varbind(OID_SYS_CONTACT, &varbinds[2]);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("构造OID失败：%s\n", api_snmp_error_to_str(err));
        api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
        return;
    }

    printf("发送GET请求到 %s:%d...（超时：%dms）\n", AGENT_IP, AGENT_PORT, TIMEOUT_MS);
    err = api_snmp_get(AGENT_IP, AGENT_PORT, NULL, varbinds, 3, TIMEOUT_MS, NULL);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("GET请求发送失败：%s\n", api_snmp_error_to_str(err));
    } else {
        uint32 start_ms = platform_get_tick_ms();
        while (platform_get_tick_ms() - start_ms < TIMEOUT_MS + 2000) {
            platform_delay_ms(10);
        }
    }

    api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
}

// SNMPv1 SET LED
void snmp_v1_set_led(int led_status) {
    ApiSnmpError err;
    char set_value_str[32];
    sprintf(set_value_str, "%d", led_status);
    char user_data[128];
    snprintf(user_data, sizeof(user_data), "%s=%d", OID_LED_CONTROL, led_status);

    safe_snmp_init(); // 替换原有api_snmp_init()
    api_snmp_register_response_callback(set_response_callback);

    ApiSnmpAgentConfig agent_cfg = {.version=API_SNMP_VERSION_1, .community=COMMUNITY_PRIVATE};
    err = api_snmp_agent_create(AGENT_IP, AGENT_PORT, &agent_cfg);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("创建Agent失败：%s\n", api_snmp_error_to_str(err));
        return;
    }

    ApiSnmpVarBind varbind;
    err = api_snmp_create_set_varbind(OID_LED_CONTROL, "int", set_value_str, &varbind);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("构造SET OID失败：%s\n", api_snmp_error_to_str(err));
        api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
        return;
    }

    printf("发送SET请求到 %s:%d... OID：%s = %d\n",
           AGENT_IP, AGENT_PORT, OID_LED_CONTROL, led_status);
    err = api_snmp_set(AGENT_IP, AGENT_PORT, NULL, &varbind, 1, TIMEOUT_MS, user_data);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("SET请求发送失败：%s\n", api_snmp_error_to_str(err));
    } else {
        uint32 start_ms = platform_get_tick_ms();
        while (platform_get_tick_ms() - start_ms < TIMEOUT_MS) {
            platform_delay_ms(10);
        }
    }

    api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
}

// SNMPv2c GET-NEXT（保留原有）
void snmp_v2c_getnext(void) {
    ApiSnmpError err;
    safe_snmp_init();
    api_snmp_register_response_callback(response_callback);

    ApiSnmpAgentConfig agent_cfg = {.version=API_SNMP_VERSION_2C, .community=COMMUNITY_PUBLIC};
    err = api_snmp_agent_create(AGENT_IP, AGENT_PORT, &agent_cfg);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("创建Agent失败：%s\n", api_snmp_error_to_str(err));
        return;
    }

    ApiSnmpVarBind varbind;
    err = api_snmp_create_get_varbind(OID_MIB_TREE, &varbind);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("构造OID失败：%s\n", api_snmp_error_to_str(err));
        api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
        return;
    }

    printf("发送GET-NEXT请求到 %s:%d... 遍历OID：%s\n",
           AGENT_IP, AGENT_PORT, OID_MIB_TREE);
    err = api_snmp_get_next(AGENT_IP, AGENT_PORT, NULL, &varbind, 1, TIMEOUT_MS, NULL);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("GET-NEXT请求发送失败：%s\n", api_snmp_error_to_str(err));
    } else {
        uint32 start_ms = platform_get_tick_ms();
        while (platform_get_tick_ms() - start_ms < TIMEOUT_MS) {
            platform_delay_ms(10);
        }
    }

    api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
}

// SNMPv2c GET-BULK（保留原有）
void snmp_v2c_bulkwalk(void) {
    ApiSnmpError err;
    uint8 non_repeaters = 0;
    uint8 max_repetitions = 5;

    safe_snmp_init();
    api_snmp_register_response_callback(response_callback);

    ApiSnmpAgentConfig agent_cfg = {.version=API_SNMP_VERSION_2C, .community=COMMUNITY_PUBLIC};
    err = api_snmp_agent_create(AGENT_IP, AGENT_PORT, &agent_cfg);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("创建Agent失败：%s\n", api_snmp_error_to_str(err));
        return;
    }

    ApiSnmpVarBind varbind;
    err = api_snmp_create_get_varbind(OID_MIB_TREE, &varbind);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("构造OID失败：%s\n", api_snmp_error_to_str(err));
        api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
        return;
    }

    printf("发送GET-BULK请求到 %s:%d... OID：%s，非重复次数：%d，最大重复次数：%d\n",
           AGENT_IP, AGENT_PORT, OID_MIB_TREE, non_repeaters, max_repetitions);
    err = api_snmp_get_bulk(AGENT_IP, AGENT_PORT, NULL, non_repeaters, max_repetitions,
                           &varbind, 1, TIMEOUT_MS, NULL);
    if (err != API_SNMP_ERR_SUCCESS) {
        printf("GET-BULK请求发送失败：%s\n", api_snmp_error_to_str(err));
    } else {
        uint32 start_ms = platform_get_tick_ms();
        while (platform_get_tick_ms() - start_ms < TIMEOUT_MS) {
            platform_delay_ms(10);
        }
    }

    api_snmp_agent_remove(AGENT_IP, AGENT_PORT);
}

// ==================== 后台监听线程（新增核心） ====================
#ifdef WIN32
THREAD_FUNC listen_thread_proc(LPVOID lpParam) {
#else
THREAD_FUNC listen_thread_proc(void* arg) {
#endif
    (void)lpParam;
    safe_snmp_init();
    api_snmp_register_trap_callback(trap_callback, NULL);

    // 启动TRAP监听
    ApiSnmpError err = api_snmp_start_trap_listener(TRAP_LISTEN_PORT);
    if (err != API_SNMP_ERR_SUCCESS) {
        // 【修改4】替换📢为[INFO]
        printf("\n [INFO] 启动TRAP监听失败：%s（错误码：%d）\n", api_snmp_error_to_str(err), err);
        printf("> "); fflush(stdout);
        g_listen_running = BOOL8_FALSE;
        return 0;
    }

    // 【修改5】替换📢为[INFO]
    printf("\n[INFO] TRAP/INFORM监听已启动（0.0.0.0:%d），等待接收消息...\n", TRAP_LISTEN_PORT);
    printf("> "); fflush(stdout);

    // 后台监听循环
    while (g_listen_running && !g_exit_flag) {
        platform_delay_ms(10); // 降低CPU占用
    }

    // 停止监听
    api_snmp_stop_trap_listener();
    printf("\n TRAP监听已停止\n");
    g_listen_running = BOOL8_FALSE;
    return 0;
}

// ==================== 启动/停止监听（新增） ====================
static void start_listen(void) {
    if (g_listen_running) {
        // 【修改6】替换⚠️为[WARN]
        printf("\n[WARN] TRAP监听已在运行中\n");
        printf("> "); fflush(stdout);
        return;
    }
    g_listen_running = BOOL8_TRUE;
    g_listen_thread = CREATE_THREAD(listen_thread_proc, NULL);
    if (!g_listen_thread) {
        printf("\n 创建监听线程失败\n");
        printf("> "); fflush(stdout);
        g_listen_running = BOOL8_FALSE;
    }
}

static void stop_listen(void) {
    if (!g_listen_running) {
        // 【修改7】替换⚠️为[WARN]
        printf("\n[WARN] TRAP监听未运行\n");
        printf("> "); fflush(stdout);
        return;
    }
    g_listen_running = BOOL8_FALSE;
    WAIT_THREAD(g_listen_thread);
    CLOSE_THREAD(g_listen_thread);
    printf("> "); fflush(stdout);
}

// ==================== 单窗口交互模式（核心修改） ====================
void interactive_mode(void) {
    char input_cmd[64];
    safe_snmp_init();
    api_snmp_register_trap_callback(trap_callback, NULL);

    // 欢迎信息
    printf("========================================\n");
    // 【修改8】替换📌为[NOTE]
    printf("[NOTE] SNMP 单窗口交互模式\n");
    printf("支持命令：\n");
    printf("  start-listen  - 启动后台TRAP监听\n");
    printf("  stop-listen   - 停止后台TRAP监听\n");
    printf("  get           - SNMPv2c GET（读取系统信息）\n");
    printf("  set-led1      - SNMPv1 SET LED=1（触发Inform）\n");
    printf("  set-led0      - SNMPv1 SET LED=0（触发Trap）\n");
    printf("  getnext       - SNMPv2c GET-NEXT（遍历MIB）\n");
    printf("  bulkwalk      - SNMPv2c GET-BULK（批量遍历）\n");
    printf("  exit          - 退出程序\n");
    printf("========================================\n");

    // 主交互循环
    while (!g_exit_flag) {
        printf("> "); fflush(stdout); // 强制显示提示符
        if (fgets(input_cmd, sizeof(input_cmd), stdin) == NULL) break;
        
        // 处理输入
        input_cmd[strcspn(input_cmd, "\n")] = '\0';
        if (strlen(input_cmd) == 0) continue;

        // 解析命令
        if (strcmp(input_cmd, "start-listen") == 0) {
            start_listen();
        } else if (strcmp(input_cmd, "stop-listen") == 0) {
            stop_listen();
        } else if (strcmp(input_cmd, "get") == 0) {
            snmp_v2c_get();
            printf("> "); fflush(stdout);
        } else if (strcmp(input_cmd, "set-led1") == 0) {
            snmp_v1_set_led(1);
            printf("> "); fflush(stdout);
        } else if (strcmp(input_cmd, "set-led0") == 0) {
            snmp_v1_set_led(0);
            printf("> "); fflush(stdout);
        } else if (strcmp(input_cmd, "getnext") == 0) {
            snmp_v2c_getnext();
            printf("> "); fflush(stdout);
        } else if (strcmp(input_cmd, "bulkwalk") == 0) {
            snmp_v2c_bulkwalk();
            printf("> "); fflush(stdout);
        } else if (strcmp(input_cmd, "exit") == 0) {
            g_exit_flag = BOOL8_TRUE;
            stop_listen();
    // ★ 新增：停止1ms定时器
    g_timer_running = BOOL8_FALSE;
    WAIT_THREAD(g_timer_thread); // 等待线程退出
    api_snmp_destroy();
            g_snmp_inited = BOOL8_FALSE;
            printf("\n退出交互模式...\n");
        } else {
            printf("\n 无效命令！\n");
            printf("> "); fflush(stdout);
        }
    }
}

// ==================== 帮助信息（保留原有） ====================
void print_help(void) {
    printf("========================================\n");
    printf("SNMP API 测试程序（单窗口模式）\n");
    printf("支持平台：Linux/Windows/STM32\n");
    printf("========================================\n");
    printf("用法1：./snmp_api_demo （无参数，进入单窗口交互模式）\n");
    printf("用法2：./snmp_api_demo -help （显示帮助）\n");
    printf("交互模式支持命令：\n");
    printf("  start-listen  - 启动后台TRAP监听\n");
    printf("  stop-listen   - 停止后台TRAP监听\n");
    printf("  get           - SNMPv2c GET（读取系统信息）\n");
    printf("  set-led1      - SNMPv1 SET LED=1（触发Inform）\n");
    printf("  set-led0      - SNMPv1 SET LED=0（触发Trap）\n");
    printf("  getnext       - SNMPv2c GET-NEXT（遍历MIB）\n");
    printf("  bulkwalk      - SNMPv2c GET-BULK（批量遍历）\n");
    printf("  exit          - 退出程序\n");
    printf("========================================\n");
}

// ==================== 主函数（入口，最小修改） ====================
int main(int argc, char* argv[]) {
    // 平台初始化
    PLATFORM_INIT();

    // 注册退出信号
#ifdef WIN32
    SetConsoleCtrlHandler(console_handler, TRUE);
#elif defined(LINUX)
    signal(SIGINT, sigint_handler);
#endif

    // 欢迎信息
    printf("========================================\n");
    printf("版权所有 (C) 芯祥联科技 All Rights Reserved.\n");
#ifdef WIN32
    printf("当前平台：Windows (MinGW)\n");
#elif defined(LINUX)
    printf("当前平台：Linux\n");
#elif defined(STM32)
    printf("当前平台：STM32 Cortex-M4\n");
#endif
    printf("========================================\n\n");

    // 命令行逻辑
    if (argc == 1) {
        interactive_mode(); // 进入单窗口交互模式
    } else if (argc == 2 && strcmp(argv[1], "-help") == 0) {
        print_help();
    } else {
        printf(" 无效参数！使用 -help 查看帮助\n");
    }

    // 平台清理
    PLATFORM_CLEANUP();
    return 0;
}
