﻿#ifndef NET_INTERFACE_H
#define NET_INTERFACE_H

#include "type.h"          // 基础类型（uint8/uint32等）
#include "nms_snmp_api.h"  // 错误码类型

#ifdef __cplusplus
extern "C" {
#endif


#define MAX_MSG_LEN 1024
//#define API_IP_LEN 16

// 网络接口类型（业务枚举，无实现细节）
typedef enum {
    NET_IFACE_ETH = 0,
    NET_IFACE_ES,
    NET_IFACE_MAX
} NetInterfaceType;

// 接收数据节点（仅暴露业务字段，无同步机制）
typedef struct NetRecvNode {
    uint8 data[MAX_MSG_LEN];
    uint32 len;
    char src_ip[API_IP_LEN];
    uint16 src_port;
} NetRecvNode;  // 注意：移除链表指针，上层无需关心节点关系

// 网络接口抽象（函数指针封装所有细节）
typedef struct {
    NetInterfaceType type;
    ApiSnmpError (*net_send)(const char* ip, uint16 port, const uint8* data, uint32 len);
    ApiSnmpError (*net_recv)(uint8* buf, uint32 buf_len, uint32* recv_len, char* src_ip, uint16* src_port);
    void (*destroy)(void);
    uint16 local_port;
} NetInterface;

// 接收回调函数（上层层无需关心队列队列如何唤醒）
typedef void (*NetRecvCallback)(void* user_data);

// --------------- 仅暴露接口（仅暴露业务操作，无同步机制）---------------
// 初始化/销毁网络模块
ApiSnmpError net_interface_init(void);
void net_interface_deinit(void);

// 网络类型切换
ApiSnmpError net_interface_set_current_type(NetInterfaceType type);
NetInterfaceType net_interface_get_current_type(void);

// 数据发送/接收（ETH/ES统一接口）
ApiSnmpError net_interface_send_to(int32 sock_fd, const char* ip, uint16 port, const uint8* data, uint32 len);

//ApiSnmpError net_interface_recv_from(int32 sock_fd, uint8* buf, uint32 buf_len, char* src_ip, uint16* src_port, uint32* recv_len);



// 队列管理（仅暴露入队/出队，不暴露内部同步）
ApiSnmpError net_interface_enqueue(const NetRecvNode* node);
ApiSnmpError net_interface_dequeue(NetRecvNode* node, uint32 timeout);
ApiSnmpError net_interface_bind(int32 sock_fd, uint16 port);
void net_interface_close(int32 sock_fd);
int32 net_interface_create_udp_socket(void);

// 注册回调（上层层层无需需关心队列如何触发回调）
//void net_interface_register_recv_callback(NetRecvCallback callback, void* user_data);

// 内部接口（供具体实现模块注册）
ApiSnmpError net_interface_register(const NetInterface* iface);

#ifdef __cplusplus
}
#endif

#endif  // NET_INTERFACE_H
