﻿/***************************************************************************
 * File Name: platform.c
 * Description: 跨平台通用逻辑实现（路由到具体平台的实现）
 * Author: ZXY
 * Company: 芯祥联科技
 * Date Created: 2025-10-22
 * Version: 1.5（修复64位指针截断问题）
 ***************************************************************************/
#include "type.h"
#include "platform.h"
#include "agent_log.h"
#include <string.h>
#include <stdint.h>  // 新增：用于指针完整性检查


// -------------------------- 平台特有实现声明（保持不变） --------------------------
#ifdef WIN32
// Windows平台新增接口声明
extern uint32 platform_thread_id_win32(void);
extern PlatformTime platform_get_utc_seconds_win32(void);
extern PlatformTime platform_get_utc_ms_win32(void);
extern bool8 platform_timer_start_win32(PlatformTimer timer);
extern bool8 platform_timer_stop_win32(PlatformTimer timer);
//extern void* platform_malloc_win32(uint32 size);
extern void* platform_calloc_win32(uint32 num, uint32 size);
extern void platform_free_win32(void* ptr);
extern bool8 platform_rand_generate_crypto_win32(uint8* buf, uint32 len);
extern bool8 platform_file_exists_win32(const char* path);
extern uint32 platform_get_file_size_win32(const char* path);
extern uint32 platform_memset_s_win32(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len);
extern int32 platform_strcmp_win32(const char* str1, const char* str2);
extern char* platform_strchr_win32(const char* str, int32 ch);
extern void platform_log_output_win32(const char* module, const char* fmt, ...);

#elif defined(LINUX)
// Linux平台新增接口声明
extern uint32 platform_thread_id_linux(void);
extern PlatformTime platform_get_utc_seconds_linux(void);
extern PlatformTime platform_get_utc_ms_linux(void);
extern bool8 platform_timer_start_linux(PlatformTimer timer);
extern bool8 platform_timer_stop_linux(PlatformTimer timer);
extern void* platform_malloc_linux(uint32 size);
extern void* platform_calloc_linux(uint32 num, uint32 size);
extern void platform_free_linux(void* ptr);
extern bool8 platform_rand_generate_crypto_linux(uint8* buf, uint32 len);
extern bool8 platform_file_exists_linux(const char* path);
extern uint32 platform_get_file_size_linux(const char* path);
extern uint32 platform_memset_s_linux(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len);
extern int32 platform_strcmp_linux(const char* str1, const char* str2);
extern char* platform_strchr_linux(const char* str, int32 ch);
extern void platform_log_output_linux(const char* module, const char* fmt, ...);

#elif defined(STM32)
// STM32平台新增接口声明
extern uint32 platform_thread_id_stm32(void);
extern PlatformTime platform_get_utc_seconds_stm32(void);
extern PlatformTime platform_get_utc_ms_stm32(void);
extern bool8 platform_timer_start_stm32(PlatformTimer timer);
extern bool8 platform_timer_stop_stm32(PlatformTimer timer);
extern void* platform_malloc_stm32(uint32 size);
extern void* platform_calloc_stm32(uint32 num, uint32 size);
extern void platform_free_stm32(void* ptr);
extern bool8 platform_rand_generate_crypto_stm32(uint8* buf, uint32 len);
extern bool8 platform_file_exists_stm32(const char* path);
extern uint32 platform_get_file_size_stm32(const char* path);
extern uint32 platform_memset_s_stm32(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len);
extern int32 platform_strcmp_stm32(const char* str1, const char* str2);
extern char* platform_strchr_stm32(const char* str, int32 ch);
extern void platform_log_output_stm32(const char* module, const char* fmt, ...);

#else
//#warning "Unsupported platform! Some functions may be unavailable."
#endif


// -------------------------- 通用接口实现（补充指针校验） --------------------------
// 线程接口（补充线程ID）
//uint32 platform_thread_id(void) {
//#ifdef WIN32
//    return platform_thread_id_win32();
//#elif defined(LINUX)
//    return platform_thread_id_linux();
//#elif defined(STM32)
//    return platform_thread_id_stm32();
//#else
//    return 0;  // 未知平台默认返回0
//#endif
//}
// -------------------------- 新增：指针完整性检查工具函数 --------------------------
/**
 * 检查指针是否为合法64位地址（未被截断）
 * @param ptr 待检查的指针
 * @return 合法返回TRUE，否则FALSE
 */
static bool8 is_valid_pointer(void* ptr) {
    if (!ptr) return BOOL8_TRUE;  // NULL指针视为合法

#ifdef _WIN64
    // 64位环境：检查高32位是否为0xFFFFFFFF（截断特征）
    uintptr_t addr = (uintptr_t)ptr;
    if ((addr & 0xFFFFFFFF00000000) == 0xFFFFFFFF00000000) {
        agent_log_error("Invalid pointer (truncated): %p", ptr);
        return BOOL8_FALSE;
    }
#endif
    return BOOL8_TRUE;
}
// -------------------------- 新增：跨平台大小端转换函数实现 --------------------------
void platform_cpu_to_be32(uint8* out, uint32 value) {
    if (!out) return;
    // 大端序：高位字节在前
    out[0] = (uint8)(value >> 24);
    out[1] = (uint8)(value >> 16);
    out[2] = (uint8)(value >> 8);
    out[3] = (uint8)(value & 0xFF);
}

uint32 platform_be32_to_cpu(const uint8* in) {
    if (!in) return 0;
    // 大端字节数组转主机序：高位字节对应高地址
    return (uint32)((uint32)in[0] << 24) | 
           (uint32)((uint32)in[1] << 16) | 
           (uint32)((uint32)in[2] << 8)  | 
           (uint32)in[3];
}

void platform_cpu_to_be16(uint8* out, uint16 value) {
    if (!out) return;
    // 大端序：高位字节在前
    out[0] = (uint8)(value >> 8);
    out[1] = (uint8)(value & 0xFF);
}

uint16 platform_be16_to_cpu(const uint8* in) {
    if (!in) return 0;
    // 大端字节数组转主机序：高位字节对应高地址
    return (uint16)((uint16)in[0] << 8) | (uint16)in[1];
}

// 时间接口（补充UTC时间）
PlatformTime platform_get_utc_seconds(void) {
#ifdef WIN32
    return platform_get_utc_seconds_win32();
#elif defined(LINUX)
    return platform_get_utc_seconds_linux();
#elif defined(STM32)
    return platform_get_utc_seconds_stm32();
#else
    return 0;  // 未知平台默认返回0
#endif
}

//PlatformTime platform_get_utc_ms(void) {
//#ifdef WIN32
//    return platform_get_utc_ms_win32();
//#elif defined(LINUX)
//    return platform_get_utc_ms_linux();
//#elif defined(STM32)
//    return platform_get_utc_ms_stm32();
//#else
//    return 0;  // 未知平台默认返回0
//#endif
//}

// 定时器接口（补充启停，新增指针校验）
//bool8 platform_timer_start(PlatformTimer timer) {
//    // 新增：检查定时器指针完整性
//    if (!is_valid_pointer(timer)) {
//        agent_log_error("platform_timer_start: 无效定时器指针（已截断）");
//        return BOOL8_FALSE;
//    }

//#ifdef WIN32
//    return platform_timer_start_win32(timer);
//#elif defined(LINUX)
//    return platform_timer_start_linux(timer);
//#elif defined(STM32)
//    return platform_timer_start_stm32(timer);
//#else
//    return BOOL8_FALSE;  // 未知平台返回失败
//#endif
//}

//bool8 platform_timer_stop(PlatformTimer timer) {
//    // 新增：检查定时器指针完整性
//    if (!is_valid_pointer(timer)) {
//        agent_log_error("platform_timer_stop: 无效定时器指针（已截断）");
//        return BOOL8_FALSE;
//    }

//#ifdef WIN32
//    return platform_timer_stop_win32(timer);
//#elif defined(LINUX)
//    return platform_timer_stop_linux(timer);
//#elif defined(STM32)
//    return platform_timer_stop_stm32(timer);
//#else
//    return BOOL8_FALSE;  // 未知平台返回失败
//#endif
//}

// 内存接口（补充基础分配）
void* platform_malloc(uint32 size) {
#ifdef WIN32
    return platform_malloc_win32(size);
#elif defined(LINUX)
    return platform_malloc_linux(size);
#elif defined(STM32)
    return platform_malloc_stm32(size);
#else
    return malloc(size);  // 未知平台 fallback 到标准库
#endif
}

//void* platform_calloc(uint32 num, uint32 size) {
//#ifdef WIN32
//    return platform_calloc_win32(num, size);
//#elif defined(LINUX)
//    return platform_calloc_linux(num, size);
//#elif defined(STM32)
//    return platform_calloc_stm32(num, size);
//#else
//    return calloc(num, size);  // 未知平台 fallback 到标准库
//#endif
//}

void platform_free(void* ptr) {
    if (!ptr) return;
    // 新增：释放前检查指针完整性（避免释放非法地址）
    if (!is_valid_pointer(ptr)) {
        agent_log_error("platform_free: 尝试释放无效指针（已截断）：%p", ptr);
        return;
    }

#ifdef WIN32
    platform_free_win32(ptr);
#elif defined(LINUX)
    platform_free_linux(ptr);
#elif defined(STM32)
    platform_free_stm32(ptr);
#else
    free(ptr);  // 未知平台 fallback 到标准库
#endif
}

// 随机数接口（补充加密级）
bool8 platform_rand_generate_crypto(uint8* buf, uint32 len) {
    if (!buf || len == 0) return BOOL8_FALSE;
#ifdef WIN32
    return platform_rand_generate_crypto_win32(buf, len);
#elif defined(LINUX)
    return platform_rand_generate_crypto_linux(buf, len);
#elif defined(STM32)
    return platform_rand_generate_crypto_stm32(buf, len);
#else
    // 未知平台使用普通随机数（安全性较低）
    return platform_rand_generate(buf, len);
#endif
}

// 文件系统接口
//bool8 platform_file_exists(const char* path) {
//    if (!path) return BOOL8_FALSE;
//#ifdef WIN32
//    return platform_file_exists_win32(path);
//#elif defined(LINUX)
//    return platform_file_exists_linux(path);
//#elif defined(STM32)
//    return platform_file_exists_stm32(path);
//#else
//    return BOOL8_FALSE;  // 未知平台默认返回不存在
//#endif
//}

//uint32 platform_get_file_size(const char* path) {
//    if (!path) return 0;
//#ifdef WIN32
//    return platform_get_file_size_win32(path);
//#elif defined(LINUX)
//    return platform_get_file_size_linux(path);
//#elif defined(STM32)
//    return platform_get_file_size_stm32(path);
//#else
//    return 0;  // 未知平台默认返回0
//#endif
//}

// 字符串/内存操作（补充安全函数）
//uint32 platform_memset_s(void* dest, uint32 dest_max_len, uint8 value, uint32 set_len) {
//    if (!dest || dest_max_len == 0) return 0;
//    uint32 actual_len = (set_len < dest_max_len) ? set_len : dest_max_len;
//#ifdef WIN32
//    return platform_memset_s_win32(dest, dest_max_len, value, actual_len);
//#elif defined(LINUX)
//    return platform_memset_s_linux(dest, dest_max_len, value, actual_len);
//#elif defined(STM32)
//    return platform_memset_s_stm32(dest, dest_max_len, value, actual_len);
//#else
//    memset(dest, value, actual_len);
//    return actual_len;
//#endif
//}

int32 platform_strcmp(const char* str1, const char* str2) {
    if (!str1 || !str2) return (str1 == str2) ? 0 : (str1 ? 1 : -1);
#ifdef WIN32
    return platform_strcmp_win32(str1, str2);
#elif defined(LINUX)
    return platform_strcmp_linux(str1, str2);
#elif defined(STM32)
    return platform_strcmp_stm32(str1, str2);
#else
    return strcmp(str1, str2);  // 未知平台 fallback 到标准库
#endif
}

uint32 platform_strlen(const char* str) {
    if (str == NULL) {
        return 0;  // 空指针安全处理
    }
     return strlen(str);
}

//char* platform_strchr(const char* str, int32 ch) {
//    if (!str) return NULL;
//#ifdef WIN32
//    return platform_strchr_win32(str, ch);
//#elif defined(LINUX)
//    return platform_strchr_linux(str, ch);
//#elif defined(STM32)
//    return platform_strchr_stm32(str, ch);
//#else
//    return strchr(str, ch);  // 未知平台 fallback 到标准库
//#endif
//}


// -------------------------- 原有接口路由（补充指针校验） --------------------------
// 线程接口
PlatformThread platform_thread_create(PlatformThreadFunc func, void* arg) {
    // 新增：检查回调函数指针完整性
    if (!is_valid_pointer((void*)func)) {
        agent_log_error("platform_thread_create: 无效回调函数指针（已截断）");
        return NULL;
    }

#ifdef WIN32
    return platform_thread_create_win32(func, arg);
#elif defined(LINUX)
    return platform_thread_create_linux(func, arg);
#elif defined(STM32)
    return platform_thread_create_stm32(func, arg);
#else
	//裸机环境可以在LOOP中调用func();
    return NULL;
#endif
}

void platform_thread_join(PlatformThread thread) {
	//线程销毁
    // 新增：检查线程指针完整性
    if (!is_valid_pointer(thread)) {
        agent_log_error("platform_thread_join: 无效线程指针（已截断）：%p", thread);
        return;
    }

#ifdef WIN32
    platform_thread_join_win32(thread);
#elif defined(LINUX)
    platform_thread_join_linux(thread);
#elif defined(STM32)
    platform_thread_join_stm32(thread);
#else
	//裸机环境可以为空。
#endif
}

//void platform_thread_detach(PlatformThread thread) {
//    // 新增：检查线程指针完整性
//    if (!is_valid_pointer(thread)) {
//        agent_log_error("platform_thread_detach: 无效线程指针（已截断）：%p", thread);
//        return;
//    }

//#ifdef WIN32
//    platform_thread_detach_win32(thread);
//#elif defined(LINUX)
//    platform_thread_detach_linux(thread);
//#elif defined(STM32)
//    platform_thread_detach_stm32(thread);
//#endif
//}

// 互斥锁接口
PlatformMutex platform_mutex_create(void) {
#ifdef WIN32
    return platform_mutex_create_win32();
#elif defined(LINUX)
    return platform_mutex_create_linux();
#elif defined(STM32)
    return platform_mutex_create_stm32();
#else
	//裸机环境可以为空
    return NULL;
#endif
}

void platform_mutex_lock(PlatformMutex mutex) {
    // 新增：检查互斥锁指针完整性
    if (!is_valid_pointer(mutex)) {
        agent_log_error("platform_mutex_lock: 无效互斥锁指针（已截断）：%p", mutex);
        return;
    }

#ifdef WIN32
    platform_mutex_lock_win32(mutex);
#elif defined(LINUX)
    platform_mutex_lock_linux(mutex);
#elif defined(STM32)
    platform_mutex_lock_stm32(mutex);
#else
		//裸机环境可以为空
#endif
}

void platform_mutex_unlock(PlatformMutex mutex) {
    // 新增：检查互斥锁指针完整性
    if (!is_valid_pointer(mutex)) {
        agent_log_error("platform_mutex_unlock: 无效互斥锁指针（已截断）：%p", mutex);
        return;
    }

#ifdef WIN32
    platform_mutex_unlock_win32(mutex);
#elif defined(LINUX)
    platform_mutex_unlock_linux(mutex);
#elif defined(STM32)
    platform_mutex_unlock_stm32(mutex);
#else
	//裸机环境可以为空

#endif
}

void platform_mutex_destroy(PlatformMutex mutex) {
    // 新增：检查互斥锁指针完整性
    if (!is_valid_pointer(mutex)) {
        agent_log_error("platform_mutex_destroy: 无效互斥锁指针（已截断）：%p", mutex);
        return;
    }

#ifdef WIN32
    platform_mutex_destroy_win32(mutex);
#elif defined(LINUX)
    platform_mutex_destroy_linux(mutex);
#elif defined(STM32)
    platform_mutex_destroy_stm32(mutex);
#else
	//裸机环境可以为空

#endif
}




int32 platform_sem_timedwait(PlatformSem sem, uint32 timeout_ms) {
    // 新增：检查信号量指针完整性
    if (!is_valid_pointer(sem)) {
        agent_log_error("platform_sem_timedwait: 无效信号量指针（已截断）：%p", sem);
        return -1;
    }

#ifdef WIN32
    return platform_sem_timedwait_win32(sem, timeout_ms);
#elif defined(LINUX)
    return platform_sem_timedwait_linux(sem, timeout_ms);
#elif defined(STM32)
    return platform_sem_timedwait_stm32(sem, timeout_ms);
#else
    return -1;
#endif
}




// 时间接口
uint32 platform_get_tick_ms(void) {
#ifdef WIN32
    return platform_get_tick_ms_win32();
#elif defined(LINUX)
    return platform_get_tick_ms_linux();
#elif defined(STM32)
    return platform_get_tick_ms_stm32();
#else
	//产生一个MS级别的tick值
    return 0;
#endif
}

//uint64 platform_get_tick_us(void) {
//#ifdef WIN32
//    return platform_get_tick_us_win32();
//#elif defined(LINUX)
//    return platform_get_tick_us_linux();
//#elif defined(STM32)
//    return platform_get_tick_us_stm32();
//#else
//    return 0;
//#endif
//}



// 定时器接口（补充指针校验）
//PlatformTimer platform_timer_create(PlatformTimerFunc func, void* arg, uint32 period_ms) {
//    // 新增：检查回调函数指针完整性
//    if (!is_valid_pointer((void*)func)) {
//        agent_log_error("platform_timer_create: 无效回调函数指针（已截断）");
//        return NULL;
//    }

//#ifdef WIN32
//    return platform_timer_create_win32(func, arg, period_ms);
//#elif defined(LINUX)
//    return platform_timer_create_linux(func, arg, period_ms);
//#elif defined(STM32)
//    return platform_timer_create_stm32(func, arg, period_ms);
//#else
//    return NULL;
//#endif
//}

//void platform_timer_destroy(PlatformTimer timer) {
//    // 新增：检查定时器指针完整性
//    if (!is_valid_pointer(timer)) {
//        agent_log_error("platform_timer_destroy: 无效定时器指针（已截断）：%p", timer);
//        return;
//    }

//#ifdef WIN32
//    platform_timer_destroy_win32(timer);
//#elif defined(LINUX)
//    platform_timer_destroy_linux(timer);
//#elif defined(STM32)
//    platform_timer_destroy_stm32(timer);
//#endif
//}

// 内存接口
//void* platform_malloc_aligned(uint32 size, uint32 align) {
//#ifdef WIN32
//    return platform_malloc_aligned_win32(size, align, PLATFORM_MEM_DEFAULT);  // 补充参数
//#elif defined(LINUX)
//    return platform_malloc_aligned_linux(size, align, PLATFORM_MEM_DEFAULT);
//#elif defined(STM32)
//    return platform_malloc_aligned_stm32(size, align, PLATFORM_MEM_DEFAULT);
//#else
//    return NULL;  // 未知平台返回NULL
//#endif
//}

//void platform_free_aligned(void* ptr) {
//    // 新增：检查对齐内存指针完整性
//    if (!is_valid_pointer(ptr)) {
//        agent_log_error("platform_free_aligned: 无效对齐内存指针（已截断）：%p", ptr);
//        return;
//    }

//#ifdef WIN32
//    platform_free_aligned_win32(ptr);
//#elif defined(LINUX)
//    platform_free_aligned_linux(ptr);
//#elif defined(STM32)
//    platform_free_aligned_stm32(ptr);
//#endif
//}

// 随机数接口
bool8 platform_rand_generate(uint8* buf, uint32 len) {
#ifdef WIN32
    return platform_rand_generate_win32(buf, len);
#elif defined(LINUX)
    return platform_rand_generate_linux(buf, len);
#elif defined(STM32)
    return platform_rand_generate_stm32(buf, len);
#else
    return BOOL8_FALSE;
#endif
}



// 通用辅助函数（保持不变）
uint32 platform_memcpy_s(void* dest, uint32 dest_max_len, const void* src, uint32 src_len) {
    if (dest == NULL || src == NULL || dest_max_len == 0) {
        return 0;
    }
    uint32 copy_len = (src_len < dest_max_len) ? src_len : dest_max_len;
    memmove(dest, src, copy_len);  // 处理缓冲区重叠
    return copy_len;
}

uint32 platform_strcpy_s(char* dest, uint32 dest_max_len, const char* src) {
    if (dest == NULL || src == NULL || dest_max_len == 0) {
        return 0;
    }
    uint32 src_len = 0;
    while (src[src_len] != '\0' && src_len < dest_max_len - 1) {
        src_len++;
    }
    memcpy(dest, src, src_len);
    dest[src_len] = '\0';  // 确保终止符
    return src_len;
}
