﻿#ifndef SNMP_BER_H
#define SNMP_BER_H

#include "type.h"
#include "snmp_protocol.h"




#define MAX_OID_CONTENT_LEN 256
// 在合适位置（如其他宏定义附近）添加
#define SNMP_MSG_FLAG_AUTH 0x01    // 认证标志位
#define SNMP_MSG_FLAG_PRIV 0x02    // 加密标志位
#define SNMP_MSG_FLAG_REPORTABLE  0x04  // Reportable 标志（必须设置）



/*********************************************************************
 * 1. ASN.1 BER基础类型定义（严格遵循SNMPv1/v2c/v3协议规范）
 * 参考：RFC 3416 4.1.1 BER Type Encoding、X.690标准
 *********************************************************************/
// 通用BER类型（SNMP核心使用）
#define SNMP_BER_TYPE_INTEGER        0x02    // 整数类型（-2^31 ~ 2^31-1）
#define SNMP_BER_TYPE_OCTET_STRING   0x04    // 字符串类型（二进制/ASCII）
#define SNMP_BER_TYPE_NULL           0x05    // NULL类型（如set请求中的无效值）
#define SNMP_BER_TYPE_OID            0x06    // 对象标识符（OID）
#define SNMP_BER_TYPE_SEQUENCE       0x30    // 序列类型（封装多个字段）
#define SNMP_BER_TYPE_IPADDRESS      0x40    // IP地址类型（4字节IPv4）
#define SNMP_BER_TYPE_COUNTER32      0x41    // 32位计数器（递增，溢出归零）
#define SNMP_BER_TYPE_GAUGE32        0x42    // 32位计量器（可增可减，保持最大值）
#define SNMP_BER_TYPE_TIMETICKS      0x43    // 时间戳（1/100秒为单位）
#define SNMP_BER_TYPE_OPAQUE         0x44    // 透明类型（封装任意二进制数据）
#define SNMP_BER_TYPE_NSAPADDRESS    0x45
#define SNMP_BER_TYPE_COUNTER64      0x46    // v2c/v3新增（RFC1905）
#define SNMP_BER_TYPE_UINTEGER32     0x47    // v2c/v3新增

// 异常类型（RFC1905）
#define SNMP_BER_TYPE_NOSUCHOBJECT   0x80
#define SNMP_BER_TYPE_NOSUCHINSTANCE 0x81
#define SNMP_BER_TYPE_ENDOFMIBVIEW   0x82

// PDU类型定义
#define SNMP_BER_PDU_GET_REQUEST     0xA0    // GET请求（获取指定OID值）
#define SNMP_BER_PDU_GETNEXT_REQUEST 0xA1    // GET-NEXT请求（遍历下一个OID）
#define SNMP_BER_PDU_RESPONSE        0xA2    // 响应PDU（返回请求结果/错误）
#define SNMP_BER_PDU_SET_REQUEST     0xA3    // SET请求（修改指定OID值）
#define SNMP_BER_PDU_TRAP_V1         0xA4    // SNMPv1陷阱（RFC1157标准值）
#define SNMP_BER_PDU_GETBULK_REQUEST 0xA5    // GET-BULK请求（批量获取，SNMPv2c新增）
#define SNMP_BER_PDU_INFORM_REQUEST  0xA6    // INFORM请求（确认型陷阱，SNMPv2c新增）
#define SNMP_BER_PDU_TRAP_V2         0xA7    // SNMPv2c陷阱（无响应需求）
#define SNMP_BER_PDU_REPORT 0xA8  // Report PDU 的 BER 类型标识（符合 RFC 3411）

#define SNMP_BER_PDU_SCOPED_PDU      0xA3    // SNMPv3作用域PDU（封装安全相关字段）
#define SNMP_BER_PDU_TRAP            SNMP_BER_PDU_TRAP_V1  // 兼容v1的陷阱别名

/*********************************************************************
 * 2. SNMP标准错误码定义（集中管理所有错误码）
 * 说明：
 * 1. 0表示成功，负数表示内部错误（编解码/协议层）
 * 2. 正数表示PDU错误状态（符合RFC规范，用于响应PDU的error-status字段）
 * 3. 避免重复定义，确保与协议层错误处理逻辑一致
 *********************************************************************/
typedef enum {
    /********** 基础成功状态 **********/
    SNMP_BER_SUCCESS = 0,                          // 操作成功（所有场景通用）

    /********** 编解码层错误 **********/
    SNMP_BER_ERR_BUFFER_TOO_SMALL = -1,            // 缓冲区不足
    SNMP_BER_ERR_INVALID_TYPE = -2,                // 无效BER类型
    SNMP_BER_ERR_INVALID_LENGTH = -3,              // 无效BER长度
    SNMP_BER_ERR_INVALID_OID = -4,                 // 无效OID
    SNMP_BER_ERR_INVALID_PARAM = -5,               // 无效参数
    SNMP_BER_ERR_INVALID_VALUE = -6,               // 无效值
    SNMP_BER_ERR_WRONG_ENCODING = -7,              // 编码格式错误

    /********** 协议层错误 **********/
    SNMP_BER_ERR_VERSION_MISMATCH = -8,            // 版本不匹配
    SNMP_BER_ERR_UNSUPPORTED_VERSION = -9,         // 不支持的版本
    SNMP_BER_ERR_NOSUCH_OBJECT = -10,              // 无此对象
    SNMP_BER_ERR_NOSUCH_INSTANCE = -11,            // 无此实例
    SNMP_BER_ERR_END_OF_MIB_VIEW = -12,            // MIB视图结束
    SNMP_BER_ERR_WRONG_TYPE = -13,                 // 类型不匹配
    SNMP_BER_ERR_WRONG_LENGTH = -14,               // 长度不合法
    SNMP_BER_ERR_WRONG_VALUE = -15,                // 数值超范围
    SNMP_BER_ERR_NOSUCH_ACCESS = -16,              // 无访问权限
    SNMP_BER_ERR_NO_CREATION = -17,                // 无法创建变量
    SNMP_BER_ERR_INCONSISTENT_VALUE = -18,         // 值冲突
    SNMP_BER_ERR_RESOURCE_UNAVAILABLE = -19,       // 资源不足
    SNMP_BER_ERR_COMMIT_FAILED = -20,              // 提交失败
    SNMP_BER_ERR_UNDO_FAILED = -21,                // 回滚失败
    SNMP_BER_ERR_AUTHORIZATION_ERROR = -22,        // 鉴权失败
    SNMP_BER_ERR_NOT_WRITABLE = -23,               // 变量不可写
    SNMP_BER_ERR_INCONSISTENT_NAME = -24,          // 变量名不匹配

    /********** SNMPv1兼容错误 **********/
    SNMP_BER_ERR_V1_NOSUCH_NAME = -25,             // v1无此名称
    SNMP_BER_ERR_V1_BAD_VALUE = -26,               // v1值错误
    SNMP_BER_ERR_V1_READ_ONLY = -27,               // v1只读
    SNMP_BER_ERR_V1_GEN_ERR = -28,                 // v1通用错误

    /********** SNMPv3安全错误 **********/
    SNMP_BER_ERR_V3_SECURITY = -29,                // v3安全验证失败
    SNMP_BER_ERR_V3_ENGINE_ID_MISMATCH = -30,      // v3引擎ID不匹配
    SNMP_BER_ERR_V3_AUTH_FAILED = -31,             // v3认证失败
    SNMP_BER_ERR_V3_PRIV_FAILED = -32,             // v3加密失败
    SNMP_BER_ERR_V3_USM_INVALID_PARAMS = -33,      // v3 USM参数无效
    SNMP_BER_ERR_V3_DECODE_PLAIN_PDU = -34,        // v3 解密后PDU解码失败
    SNMP_BER_ERR_GEN_ERR = -35,                    // 通用错误
    // --- MODIFICATION START: 新增时间窗口校验错误码 ---
    SNMP_BER_ERR_V3_NOT_IN_TIME_WINDOW = -36,      // v3 消息时间戳不在有效窗口内
    // --- MODIFICATION END ---

    /********** PDU错误状态（正数，符合RFC规范） **********/
    SNMP_ERR_NO_ERROR = 0,                         // 无错误（RFC通用）
    SNMP_ERR_TOO_BIG = 1,                          // 响应过大（RFC通用）
    SNMP_ERR_NO_SUCH_NAME = 2,                     // 无此名称（v1专用，RFC1157）
    SNMP_ERR_BAD_VALUE = 3,                        // 值错误（v1专用）
    SNMP_ERR_READ_ONLY = 4,                        // 只读（v1专用）
    SNMP_ERR_GEN_ERR = 5,                          // 通用错误（v1专用）
    SNMP_ERR_NO_ACCESS = 6,                        // 无访问权限（v2c/v3，RFC1905）
    SNMP_ERR_WRONG_TYPE = 7,                       // 类型不匹配（v2c/v3）
    SNMP_ERR_WRONG_LENGTH = 8,                     // 长度不合法（v2c/v3）
    SNMP_ERR_WRONG_ENCODING = 9,                   // 编码错误（v2c/v3）
    SNMP_ERR_WRONG_VALUE = 10,                     // 数值无效（v2c/v3）
    SNMP_ERR_NO_CREATION = 11,                     // 无法创建（v2c/v3）
    SNMP_ERR_INCONSISTENT_VALUE = 12,              // 值冲突（v2c/v3）
    SNMP_ERR_RESOURCE_UNAVAILABLE = 13,            // 资源不足（v2c/v3）
    SNMP_ERR_COMMIT_FAILED = 14,                   // 提交失败（v2c/v3）
    SNMP_ERR_UNDO_FAILED = 15,                     // 回滚失败（v2c/v3）
    SNMP_ERR_AUTHORIZATION_ERROR = 16,             // 鉴权失败（v2c/v3）
    SNMP_ERR_NOT_WRITABLE = 17,
    SNMP_ERR_INCONSISTENT_NAME = 18,
    SNMP_ERR_END_OF_MIB_VIEW = 255   /* 新增：MIB视图结束 */
} SnmpBerResult;

/*********************************************************************
 * 4. 通用编解码接口（所有 SNMP 版本共享，无版本依赖）
 *********************************************************************/
// 长度字段编解码（BER 短/长格式通用）
SnmpBerResult snmp_ber_encode_length(uint8 *buf, uint16 buf_max_len, uint16 length, uint16 *out_len);
SnmpBerResult snmp_ber_decode_length(const uint8 *data, uint16 data_len, uint16 *out_length, uint16 *out_used_len);

// 整数类型编解码（支持 int32，兼容所有版本的整数字段）
SnmpBerResult snmp_ber_encode_integer(uint8 *buf, uint16 buf_max_len, int32 value, uint16 *out_len);
SnmpBerResult snmp_ber_decode_integer(const uint8 *data, uint16 data_len, int32 *out_value, uint16 *out_used_len);

// OID 编解码（支持最大 MAX_OID_CONTENT_LEN 长度）
SnmpBerResult snmp_ber_encode_oid(uint8 *buf, uint16 buf_max_len, const SnmpOid *oid, uint16 *out_len);
SnmpBerResult snmp_ber_decode_oid(const uint8 *data, uint16 data_len, SnmpOid *out_oid, uint16 *out_used_len);

// 字符串类型编解码（支持 OctetString，兼容二进制/ASCII 数据）
SnmpBerResult snmp_ber_encode_string(uint8 *buf, uint16 buf_max_len, const uint8 *str, uint16 str_len, uint16 *out_len);
SnmpBerResult snmp_ber_decode_string(const uint8 *data, uint16 data_len, uint8 *out_str, uint16 str_max_len, 
                                    uint16 *out_str_len, uint16 *out_used_len);

// 序列头部编解码（消息/PDU/变量绑定列表的外层序列通用）
SnmpBerResult snmp_ber_encode_sequence_header(uint8 *buf, uint16 buf_max_len, uint16 seq_content_len, uint16 *out_header_len);
SnmpBerResult snmp_ber_decode_sequence_header(const uint8 *data, uint16 data_len, uint16 *out_content_len, uint16 *out_used_len);

// Counter64 编解码（SNMPv2c/v3 专用，但接口通用化，v1 调用时返回版本不支持）
SnmpBerResult snmp_ber_encode_counter64(uint8* buf, uint16 buf_max_len, uint64 value, uint16* out_len);
SnmpBerResult snmp_ber_decode_counter64(const uint8* data, uint16 data_len, uint64* out_value, uint16* out_used_len);

/*********************************************************************
 * 5. 变量绑定编解码接口（处理版本兼容性，如v1不支持Counter64）
 *********************************************************************/
SnmpBerResult snmp_ber_encode_varbind(uint8* buf, uint16 buf_max_len, 
                                     const SnmpVarBind* varbind, uint8 msg_version,
                                     uint16* out_len);
SnmpBerResult snmp_ber_decode_varbind(const uint8* data, uint16 data_len, 
                                     SnmpVarBind* out_varbind, uint8 msg_version,
                                     uint16* out_used_len);

/*********************************************************************
 * 6. 消息与 PDU 编解码接口（通用层，v1/v2c 直接使用，v3 调用扩展接口）
 *********************************************************************/
// 消息版本预提取（快速过滤未启用版本，减少无效解码）
SnmpBerResult snmp_ber_peek_version(const uint8* data, uint16 data_len, uint8* out_version, uint16* out_used);

// PDU 编解码（支持所有版本的 PDU 类型，v3 会自动适配 Report/ScopedPDU）
SnmpBerResult snmp_ber_encode_snmp_pdu(uint8 *buf, uint16 buf_max_len, const SnmpPdu *pdu, 
                                      uint8 snmp_version, uint16 *out_len);
SnmpBerResult snmp_ber_decode_snmp_pdu(const uint8 *data, uint16 data_len, uint8 snmp_version, SnmpPdu *out_pdu, uint16 *out_used_len);

// 完整消息编解码（v1/v2c 直接处理，v3 会调用 snmpv3_ber.h 中的扩展接口）
SnmpBerResult snmp_ber_encode_snmp_message(uint8 *buf, uint16 buf_max_len, const SnmpV3UsmUser* v3_user, 
                                          SnmpMessage *msg, uint32 *out_len);

SnmpBerResult snmp_ber_decode_snmp_message(const uint8 *data, uint16 data_len, SnmpMessage *out_msg, uint16 *out_used_len);
SnmpBerResult snmp_ber_encode_length(uint8 *buf, uint16 buf_max_len, uint16 length, uint16 *out_len);


#endif  // SNMP_BER_H
