﻿#ifndef SNMP_PROTOCOL_H
#define SNMP_PROTOCOL_H

#include "type.h"

//#ifndef SNMPV3_TRIAL
//#define SNMPV3_TRIAL 1  // 1=试用版（仅SHA/DES），0=完整版（全部算法）
//#endif


#define CRYPTO_SHA256_DIGEST_LEN 32
#define CRYPTO_AES_256_KEY_LEN 32
#define SURPPORT_MAX_VARBIND_COUNT 10
#define SNMP_V3_WORK_BUF_LEN 2048

/*****************************************
 * 通用常量定义（符合RFC规范）
 *****************************************/
#define MAX_VARBIND_COUNT 16        // 最大变量绑定数（兼容所有版本）
#define MAX_VALUE_LEN 300            // 最大值长度（字符串/二进制）
#define MAX_MSG_LEN 1024            // 最大消息长度（嵌入式场景常用配置）
#define MAX_OID_LEN 32              // 最大OID长度（兼容MIB-2标准）
#define MAX_ENGINE_ID_LEN 32        // 引擎ID最大长度（RFC3411标准）
#define SNMP_DEFAULT_REQUEST_ID 1   // 默认请求ID起始值

#define USM_MAX_AUTH_PARAM_BUF_LEN 32

/*****************************************
 * 协议相关长度定义（RFC合规）
 *****************************************/
#define USM_USERNAME_MAX_LEN      32    // RFC3414（USM用户名最大长度）
#define USM_AUTH_PASS_MAX_LEN     64    // 【修改】适配UTF-16LE（原32→64，避免编码后溢出）
#define USM_PRIV_PASS_MAX_LEN     64    // 【修改】同上（原32→64）
#define USM_AUTH_PARAM_LEN        12    // HMAC认证参数长度（MD5/SHA1/SHA256共用，固定12字节）
#define USM_AUTH_PARAM_LEN_SHA256_FULL 32 // 新增：SHA256FULL专属（完整32字节认证参数）
#define USM_PRIV_PARAM_LEN        16     // 加密参数长度（固定）
#define SNMP_MSG_ID_MAX           0x7FFFFFFF  // 消息ID最大值（RFC3412）
#define SNMP_MSG_FLAGS_LEN        1     // v3消息标志长度
#define SNMP_MSG_MAX_BOOOTS       0xFFFF // 引擎启动次数最大值
#define SNMP_DEFAULT_PORT         161   // 标准端口（RFC1157）
#define SNMP_TRAP_DEFAULT_PORT    162   // 陷阱端口（RFC1157）

/*****************************************
 * SNMP版本定义（严格遵循RFC）
 *****************************************/
#define SNMP_V1     0  // RFC1157（v1版本号）
#define SNMP_V2C    1  // RFC1901（v2c版本号）
#define SNMP_V3     3  // RFC3410（v3版本号）

/*****************************************
 * 操作类型枚举（与PDU类型映射）
 *****************************************/
typedef enum {
    // 标准请求类型（值=RFC定义的PDU编码）
    SNMP_OP_GET = 0,                  // RFC: GET请求PDU编码=0
    SNMP_OP_GETNEXT = 1,              // RFC: GETNEXT请求PDU编码=1
    SNMP_OP_RESPONSE = 2,             // RFC: 通用响应PDU编码=2（所有请求的回复均用此编码）
    SNMP_OP_SET = 3,                  // RFC: SET请求PDU编码=3
    SNMP_OP_GETBULK = 5,              // RFC: GETBULK请求PDU编码=5（跳过未使用的4）
    SNMP_OP_TRAP = 6,                 // RFC: TRAP v2 PDU编码=6
    SNMP_OP_INFORM = 7,               // RFC: INFORM请求PDU编码=7
    SNMP_OP_REPORT = 8,               // RFC: REPORT PDU（V3）编码=8

    // 响应类型别名（仅为代码可读性，实际编码复用SNMP_OP_RESPONSE=2，符合RFC）
    SNMP_OP_GET_RESPONSE = SNMP_OP_RESPONSE,      // GET请求的响应（实际编码=2）
    SNMP_OP_GETNEXT_RESPONSE = SNMP_OP_RESPONSE,  // GETNEXT请求的响应（实际编码=2）

    SNMP_OP_UNKNOWN = 0xFF            // 无效类型
} SnmpOperation;

/*****************************************
 * 加密算法定义（RFC3414 USM标准）
 *****************************************/
#define CRYPTO_AUTH_NONE 0  // 无认证
#define CRYPTO_AUTH_MD5  1  // HMAC-MD5（RFC3414）
#define CRYPTO_AUTH_SHA  2  // HMAC-SHA（RFC3414）
// 算法标识补充
#define CRYPTO_AUTH_SHA256  3
#define CRYPTO_AUTH_SHA256_FULL 4// 对应LIBTOM_AUTH_ALG_SHA256_256

#define CRYPTO_PRIV_NONE 0  // 无加密
#define CRYPTO_PRIV_DES  1  // DES（RFC3414）
#define CRYPTO_PRIV_AES  2  // AES（RFC3826扩展）
#define CRYPTO_PRIV_AES_192 3   // 对应LIBTOM_SNMPV3_ALG_AES_192
#define CRYPTO_PRIV_AES_256 4   // 对应LIBTOM_SNMPV3_ALG_AES_256
#define CRYPTO_PRIV_3DES 5      // 对应LIBTOM_SNMPV3_ALG_3DES_EDE

#define CRYPTO_SHA_DIGEST_LEN 20  // SHA哈希结果长度
#define CRYPTO_MD5_DIGEST_LEN 16  // 【新增】补充MD5长度，适配KDF实现


/*****************************************
 * 【新增】跨平台密码编码枚举（适配Windows/Linux）
 *****************************************/
typedef enum {
    SNMP_V3_PASS_ENC_ASCII,    // ASCII/UTF-8（Linux默认）
    SNMP_V3_PASS_ENC_UTF16LE   // UTF-16LE（Windows默认）
} SnmpV3PassEncType;

/*****************************************
 * 核心数据结构（标准兼容设计）
 *****************************************/
// OID结构（对象标识符）- 修改为uint32数组存储节点值
typedef struct {
    uint32 data[MAX_OID_LEN];  // OID节点数组（直接存储十进制值）
    uint8 len;                 // 节点数量
} SnmpOid;

// 变量绑定结构（OID+值对，RFC1905）
typedef struct {
    SnmpOid oid;              // 变量OID
    uint8 type;               // 数据类型（引用snmp_ber.h中的SNMP_BER_TYPE_*）
    union {
        int32 integer;        // 整数类型
        uint32 uinteger32;    // 无符号32位整数
        uint32 counter32;     // 32位计数器
        uint32 gauge32;       // 32位计量器
        uint32 timeticks;     // 时间戳（1/100秒）
        uint64 counter64;     // 64位计数器（v2c/v3）
        uint8 ipaddress[4];   // IPv4地址
        struct {
            uint8 data[MAX_VALUE_LEN];  // 字符串/二进制数据
            uint16 len;                  // 数据长度
        } string;
        SnmpOid oid_value;    // OID类型的值
    } value;
} SnmpVarBind;

// PDU结构（协议数据单元，RFC1905/RFC3416）
typedef struct {
    SnmpOperation op;         // 操作类型
    int32 request_id;         // 请求ID（匹配请求与响应）
    int32 error_status;       // 错误状态（0表示成功）
    int32 error_index;        // 错误索引（错误变量的位置）
    uint32 non_repeaters;     // GETBULK参数（非重复次数）
    uint32 max_repetitions;   // GETBULK参数（最大重复次数）
    struct {                  // v3上下文信息（RFC3411）
        uint8 engine_id[MAX_ENGINE_ID_LEN];
        uint16 engine_id_len;
        char context_name[MAX_VALUE_LEN];
        uint16 context_name_len;
    } v3_ctx;
    SnmpVarBind varbinds[MAX_VARBIND_COUNT];  // 变量绑定列表
    uint8 varbind_count;      // 变量绑定数量
} SnmpPdu;

// SNMPv3 USM安全参数（RFC3414）
typedef struct {
    char username[USM_USERNAME_MAX_LEN];  // 用户名
    uint8 user_len;                       // 用户名长度
    uint8 auth_proto;                     // 认证协议（CRYPTO_AUTH_*）
    uint8 priv_proto;                     // 加密协议（CRYPTO_PRIV_*）
    uint16 auth_param_offset;
    uint8 auth_param[USM_MAX_AUTH_PARAM_BUF_LEN]; // 认证参数
    uint16 auth_param_len;                 // <--- 新增：认证参数长度
    uint16 priv_param_offset;             // 加密参数在消息中的绝对偏移量（核心）
    uint8 priv_param[USM_PRIV_PARAM_LEN]; // 加密参数
    uint16 priv_param_len;                 // <--- 新增：加密参数长度
    uint32 engine_boots;                  // 引擎启动次数
    uint32 engine_time;                   // 引擎时间（秒）

	// 【新增】完整IV存储字段（一次生成，编码/加密复用）
	uint8 full_iv[32];					   // 完整IV（固定部分+随机部分，适配所有算法）
	uint8 full_iv_len;					   // 完整IV实际长度（AES=16/3DES=12/DES=8字节）


    uint8 engine_id[MAX_ENGINE_ID_LEN];   // 引擎ID
    uint16 engine_id_len;                  // 引擎ID长度
    uint8 msg_id_len;                     // msgID 字段的BER编码长度
    uint8 msg_max_size_len;               // msgMaxSize 字段的BER编码长度
    uint8 msg_flags_len;                  // msgFlags 字段的BER编码长度
    uint8 msg_security_model_len;         // msgSecurityModel 字段的BER编码长度
    uint16 usm_sequence_offset;           // 【新增】USM内部SEQUENCE的绝对偏移
} SnmpV3UsmParams;

// 【修改】AgentSnmpConfig补充跨平台字段（不删除原有字段，仅新增）
typedef struct {
    bool8 enable_v1;
    bool8 enable_v2c;
    bool8 enable_v3;              // 启用SNMPv3
    bool8 v1_counter64_truncate;  // v1中截断64位计数器（兼容处理）
    
	chart8 ro_community[32];  // 只读共同体（对应标准public）
    uint8 ro_comm_len;        // 只读共同体长度
    chart8 rw_community[32];  // 读写共同体（对应标准private）
    uint8 rw_comm_len;        // 读写共同体长度    
    
// v3安全配置
    char v3_username[USM_USERNAME_MAX_LEN];
    uint8 v3_user_len;
    char v3_auth_pass[USM_AUTH_PASS_MAX_LEN];
    uint8 v3_auth_pass_len;
    char v3_priv_pass[USM_PRIV_PASS_MAX_LEN];
    uint8 v3_priv_pass_len;
    uint8 v3_auth_proto;          // 默认认证协议
    uint8 v3_priv_proto;          // 默认加密协议
    // 【新增】跨平台关键字段
    SnmpV3PassEncType v3_pass_enc; // 密码编码格式（ASCII/UTF16LE）
    //uint32 v3_kdf_iter;            // KDF迭代次数（RFC强制1000次）
    
    // 本地引擎信息（RFC3411）
    uint8 local_engine_id[MAX_ENGINE_ID_LEN];
    uint8 local_engine_id_len;
    uint32 local_engine_boots;    // 引擎启动次数
    uint32 local_engine_time;     // 引擎运行时间（秒）

//    TransCtx *eth_trans_ctx;      // 以太网传输上下文（仅保留指针）
//    TransCtx *es_trans_ctx;       // 其他传输上下文（仅保留指针）
    
    uint16 eth_port;              // 以太网端口
    uint16 es_port;               // 其他端口
    
    uint32 poll_interval;         // 轮询间隔（秒）
	char contact[64];
	char location[64];

} AgentSnmpConfig;


// SNMP消息结构（完整消息封装）
typedef struct {
    uint8 version;                // 协议版本（SNMP_V1/V2C/V3）
    union {
        struct {                  // v1/v2c安全参数（共同体名）
            char community[32];
            uint16 comm_len;
        } v1v2c;
        SnmpV3UsmParams v3;       // v3安全参数（USM）
    } sec;                        // 安全参数联合体
    // 【新增】ScopedPDU位置信息（v3专用）
    uint16 scoped_pdu_type_offset; // ScopedPDU类型字段的绝对偏移（0x30/0x04）
    uint16 scoped_pdu_len_offset;  // ScopedPDU长度字段的绝对偏移
    uint16 scoped_pdu_val_offset;  // ScopedPDU值字段的绝对偏移
    uint16 scoped_pdu_val_len;     // ScopedPDU值字段的长度
    SnmpPdu pdu;                  // 协议数据单元
    uint8 encrypted_pdu[MAX_MSG_LEN];  // 加密的PDU（v3）
    uint16 encrypted_pdu_len;     // 加密PDU长度
    uint8 msg_flags;              // v3消息标志（认证/加密标志）
    uint32 msg_id;                // v3消息ID
    uint16 auth_param_offset;     // AuthenticationParameters在原始消息中的起始偏移量
//    const uint8* raw_msg_data;    // 指向原始消息数据的指针（鉴权时需用原始字节）
} SnmpMessage;

/*****************************************
// * SNMPv3 多用户管理扩展（新增）
// *****************************************/
// 单个 USM 用户配置
typedef struct {
    bool8 is_enabled;             // 【新增】用户启用状态（实现代码依赖）
    uint8 user_len;
    char username[USM_USERNAME_MAX_LEN];

    uint8 auth_proto;
    uint8 priv_proto;
    uint8 auth_key[CRYPTO_SHA256_DIGEST_LEN];  // 从20→32字节
    uint16 auth_key_len;
    uint8 priv_key[CRYPTO_AES_256_KEY_LEN];    // 从20→32字节
    uint16 priv_key_len;

	bool8 key_generated;			   // 密钥是否已生成
	uint32 engine_boots_at_generation; // 生成密钥时的engineBoots
	uint32 engine_time_at_generation;  // 生成密钥时的engineTime
} SnmpV3UsmUser;



/*****************************************
 * 函数声明（标准协议处理）
 *****************************************/
// OID操作（符合RFC1905字典序规则）
bool8 snmp_oid_compare(const SnmpOid* oid1, const SnmpOid* oid2);
void snmp_oid_copy(SnmpOid* dest, const SnmpOid* src);
bool8 snmp_oid_is_subtree(const SnmpOid* oid, const SnmpOid* parent);
int32 snmp_oid_compare_lexicographic(const SnmpOid* oid1, const SnmpOid* oid2);

// VarBind操作
SnmpVarBind* snmp_pdu_add_varbind(SnmpPdu* pdu);
void snmp_pdu_clear_varbinds(SnmpPdu* pdu);

// v3安全处理（RFC3414）
bool8 snmp_v3_validate_usm(const SnmpV3UsmParams* usm, const AgentSnmpConfig* config);
void snmp_v3_copy_usm(SnmpV3UsmParams* dest, const SnmpV3UsmParams* src);
void snmp_v3_update_local_engine_time(AgentSnmpConfig* cfg);

bool8 snmp_send_message(const SnmpMessage* msg, const AgentSnmpConfig* cfg);

int32 snmp_map_error_code(int32 original_err, uint8 version);
/*****************************************
 * 操作类型转换函数（协议规范映射）
 *****************************************/
/**
 * @brief 将SNMP请求操作类型转换为对应的响应操作类型（符合RFC规范）
 * @param req_op 请求操作类型（来自NMS的请求PDU类型）
 * @return 对应的响应操作类型（Agent应返回的响应PDU类型），无效类型返回SNMP_OP_UNKNOWN
 */
SnmpOperation snmp_get_response_op(SnmpOperation req_op);

uint32 snmp_generate_request_id(void);
void snmp_v3_store_engine_info(const uint8* engine_id, uint8 engine_id_len,
                              uint32 engine_boots, uint32 engine_time);

#endif  // SNMP_PROTOCOL_H
