﻿#ifndef TRANS_H
#define TRANS_H

#include "type.h"
#include "platform.h"
#include "snmp_agent_cfg.h"

// 传输类型（兼容原有定义）
typedef enum { TRANS_ETH = 0, TRANS_ES = 1, TRANS_MAX } TransType;

// IP配置模式
typedef enum {
    IP_MODE_STATIC = 0,  // 静态IP
    IP_MODE_DHCP = 1     // DHCP动态获取
} IpMode;
#define TRANS_QUEUE_DEPTH  8       // 接收队列深度（支持多消息并发）
#define SNMP_ETH_PORT      161     // ETH默认端口
#define SNMP_ES_PORT       1610    // ES默认端口

// 新增：MTU定义（硬件层限制，需与网络类型匹配）
#define TRANS_ETH_MTU      1500    // 以太网MTU（标准值）
#define TRANS_ETH_PAYLOAD  (TRANS_ETH_MTU - 28)  // UDP有效载荷（1500-IP头20-UDP头8）
#define TRANS_ES_MTU       1200    // ES协议MTU（示例值，按ARINC 664配置）
#define TRANS_ES_PAYLOAD   (TRANS_ES_MTU - 16)   // ES有效载荷（示例值）

typedef enum {
    TRANS_STATE_UNINIT = 0,
    TRANS_STATE_READY = 1,
    TRANS_STATE_ERROR = 2
} TransState;

// 接收队列结构（保留原有，用于缓存完整消息）
typedef struct {
    uint8  buf[TRANS_QUEUE_DEPTH][MAX_MSG_LEN];
    uint16 len[TRANS_QUEUE_DEPTH];
    volatile uint8 head, tail, count;  // volatile确保中断/主循环同步
} TransRecvQueue;

// 平台适配层句柄（移植时仅修改此结构成员）
typedef struct {
#ifdef STM32
    void* udp_pcb;              // LWIP UDP控制块（ETH）
    void* es_handle;            // ES协议栈句柄
    uint32 eth_irq_num;         // ETH中断号（如ETH_IRQn）
    uint32 es_irq_num;          // ES中断号
#elif defined(RISCV)
    uint32 eth_base;            // ETH控制器基地址（如0x10000000）
    uint32 es_base;             // ES控制器基地址（如0x10001000）
#else
    void* hw_specific;          // 通用平台自定义硬件句柄
#endif
} TransHwHandle;

// 统一传输上下文（架构核心，隐藏底层细节）
typedef struct {
    TransType type;             // 传输类型（ETH/ES）
    TransState state;           // 状态（未初始化/就绪/错误）
    uint16 port;                // 绑定端口
    TransRecvQueue recv_q;      // 接收队列（缓存完整消息）
    TransHwHandle hw;           // 硬件适配句柄
    void (*recv_cb)(TransType, const uint8*, uint16);  // 消息回调
    
    // NMS地址记录
    uint8 nms_ip[4];            // 存储NMS的IP地址
    uint16 nms_port;            // 存储NMS的端口号
    bool8 nms_addr_valid;
    
    // 新增：IP配置字段
    IpMode ip_mode;             // IP获取方式（静态/DHCP）
    uint8 local_ip[4];          // 静态IP地址
    uint8 gateway[4];           // 网关地址
    uint8 subnet_mask[4];       // 子网掩码
    bool8 use_dhcp;             // 是否使用DHCP
    
    // 新增：默认NMS地址
    uint8 default_nms_ip[4];    // 该传输方式对应的默认NMS IP
    
    // NMS白名单
    uint8 whitelist[NMS_WHITELIST_MAX][4];
    uint8 whitelist_count;
} TransCtx;

// 新增：统一接口（核心逻辑，与Agent层对接）
bool8 trans_init(TransType type, uint16 port, 
                void (*recv_cb)(TransType, const uint8*, uint16), 
                TransCtx* ctx);
bool8 trans_send(TransCtx* ctx, const uint8* data, uint16 len, 
                const uint8* dst_ip, uint16 dst_port);  // dst_ip为NULL时适配ES
bool8 trans_recv(TransCtx* ctx, uint8* buf, uint16* len);  // 从队列读取完整消息

// 新增：连接重置接口
bool8 trans_reset(TransCtx* ctx);

// 兼容旧接口
bool8 trans_cfg_trans(TransType type, uint16 port, TransCtx* ctx);
bool8 trans_start(TransCtx* ctx);
void trans_stop(TransCtx* ctx);
bool8 trans_send_eth(const uint8* data, uint16 len);  // 旧ETH发送接口
bool8 trans_send_es(const uint8* data, uint16 len);   // 旧ES发送接口

// 中断处理函数（硬件中断触发后调用，内部处理分片重组）
void trans_eth_irq_handler(TransCtx* ctx, const uint8* data, uint16 len, 
                           const uint8* src_ip, uint16 src_port);
void trans_es_irq_handler(TransCtx* ctx, const uint8* data, uint16 len, 
                          const uint8* src_addr, uint16 src_port);

// NMS白名单设置函数
void trans_set_nms_whitelist(TransCtx* ctx, const uint8 ips[][4], uint8 count);

// 新增：IP管理接口
bool8 trans_load_ip_config(TransCtx* ctx, const char* config_path);
bool8 trans_auto_config_ip(TransCtx* ctx);
void trans_set_default_nms_ip(TransCtx* ctx, const uint8* nms_ip);
const uint8* trans_get_local_ip(TransCtx* ctx);
const uint8* trans_get_nms_ip(TransCtx* ctx);

#endif  // TRANS_H
    
