﻿#ifndef AGENT_LOG_H
#define AGENT_LOG_H

// 1. 先包含标准库头文件（解决 va_list、errno、uint32 等依赖）
//#include <stdarg.h>   // 定义 va_list/va_start/va_end
#include "type.h"
// 2. C++ 兼容性保护（避免 C++ 编译器名字修饰导致链接错误）
#ifdef __cplusplus
extern "C" {
#endif

// 3. 包含自定义头文件（依赖标准库类型，需放在标准库之后）

// 日志级别（按严重性递增）
typedef enum {
    AGENT_LOG_DEBUG,    // 调试信息（仅开发环境）
    AGENT_LOG_INFO,     // 普通信息（如节点初始化、状态变更）
    AGENT_LOG_WARN,     // 警告（如参数越界但可恢复）
    AGENT_LOG_ERROR,    // 错误（如操作失败）
    AGENT_LOG_FATAL     // 致命错误（如系统不可用）
} AgentLogLevel;

// 日志输出目标（支持多目标同时输出）
typedef enum {
    AGENT_LOG_DEST_CONSOLE = 1 << 0,  // 控制台输出
    AGENT_LOG_DEST_FLASH   = 1 << 1,  // FLASH存储（预留）
    AGENT_LOG_DEST_NMS     = 1 << 2   // 通过MIB OID上传NMS（预留）
} AgentLogDest;

// 日志配置结构体
typedef struct {
    AgentLogLevel level;       // 日志级别阈值（只输出>=该级别的日志）
    AgentLogDest dest;         // 日志输出目标（位图组合）
    uint32 max_flash_size;     // FLASH日志最大容量（字节）
    uint16 max_log_length;     // 单条日志最大长度
} AgentLogConfig;

// 日志条目结构体（用于存储和上传）
typedef struct {
    uint32 timestamp;          // 时间戳（ms）
    AgentLogLevel level;       // 日志级别
    chart8 message[256];         // 日志内容
} AgentLogEntry;

// 初始化日志系统
void agent_log_init(const AgentLogConfig* config);

// 核心日志输出函数（支持可变参数）
void agent_log(AgentLogLevel level, const char* fmt, ...);

// 按级别输出日志的便捷宏
#define agent_log_debug(fmt, ...)  agent_log(AGENT_LOG_DEBUG, fmt, ##__VA_ARGS__)
#define agent_log_info(fmt, ...)   agent_log(AGENT_LOG_INFO, fmt, ##__VA_ARGS__)
#define agent_log_warn(fmt, ...)   agent_log(AGENT_LOG_WARN, fmt, ##__VA_ARGS__)
#define agent_log_error(fmt, ...)  agent_log(AGENT_LOG_ERROR, fmt, ##__VA_ARGS__)
#define agent_log_fatal(fmt, ...)  agent_log(AGENT_LOG_FATAL, fmt, ##__VA_ARGS__)
/**
 * 设置日志输出级别
 * @param level 目标日志级别（AGENT_LOG_* 枚举值）
 */
void agent_log_set_level(AgentLogLevel level);

// 日志管理接口（预留扩展）
uint16 agent_log_get_count(void);
bool8 agent_log_get_entry(uint16 index, AgentLogEntry* entry);
void agent_log_clear_flash(void);
void agent_log_hexdump(const char* prefix, const uint8* data, uint16 len);
void agent_log_printf(const char* format, ...);

#ifdef __cplusplus
}
#endif
#endif // AGENT_LOG_H
