﻿#ifndef AGENT_MIB_H
#define AGENT_MIB_H

#include "snmp_ber.h"
#include "snmp_protocol.h"
#include "type.h"

/*****************************************
 * 架构常量定义（统一维护，避免魔法数字）
 *****************************************/
#define MAX_MIB_NODES       32      // MIB节点最大数量（与数据库数组匹配）
#define MIB_DEFAULT_POLL    1000    // 默认采集周期（1秒）
#define MIB_NO_POLL         0       // 非周期性采集标记


/*****************************************
 * 核心枚举与结构体定义（覆盖MIB节点全属性）
 *****************************************/
// MIB节点访问权限（遵循SNMP标准访问控制）
typedef enum {
    MIB_PERM_READ_ONLY,    // 只读：支持GET/GETNEXT/GETBULK
    MIB_PERM_READ_WRITE,   // 读写：支持所有SNMP操作
    MIB_PERM_WRITE_ONLY    // 只写：仅支持SET
} MibPermission;

// MIB节点操作回调（解耦业务逻辑与MIB框架）
typedef struct {
    /**
     * @brief 读回调：获取节点实时数据
     * @param user_data [in] 自定义上下文（如硬件寄存器地址）
     * @param value [out] 输出数据（需调用方提前分配内存）
     * @return SnmpBerResult 操作结果（成功/错误码）
     */
    SnmpBerResult (*read)(void* user_data, SnmpVarBind* value);
    
    /**
     * @brief 写回调：设置节点数据
     * @param user_data [in] 自定义上下文
     * @param value [in] 待设置数据（类型需与节点匹配）
     * @return SnmpBerResult 操作结果（成功/错误码）
     */
    SnmpBerResult (*write)(void* user_data, const SnmpVarBind* value);
} MibNodeOps;

// MIB节点配置结构体（描述节点完整信息）
typedef struct {
    SnmpOid oid;                  // 节点唯一OID
    uint8 type;                   // 数据类型（SNMP_BER_TYPE_xxx，如整数、字符串）
    MibPermission perm;           // 访问权限
    uint32 poll_interval;         // 采集周期（ms，0=不周期采集）
    MibNodeOps op;                // 操作回调
    void* user_data;              // 回调上下文（透传）
    const char* name;             // 节点名称（调试用）
} MibNodeConfig;

/*****************************************
 * 架构核心函数声明（与源文件1:1匹配）
 *****************************************/
/**
 * @brief 初始化MIB模块
 * @note 系统启动时调用，初始化数据库、注册表及默认节点
 */
void agent_mib_init(void);

/**
 * @brief 单次MIB数据采集（主循环/定时器触发）
 * @note 按节点周期调用read回调，更新MIB数据库
 */
void agent_mib_coll_process(void);

/**
 * @brief 处理SNMP GET请求
 * @param oid [in] 目标节点OID
 * @param varbind [out] 输出节点数据
 * @return SnmpBerResult 操作结果
 */
SnmpBerResult agent_mib_get(const SnmpOid* oid, SnmpVarBind* varbind);

/**
 * @brief 处理SNMP SET请求
 * @param oid [in] 目标节点OID
 * @param varbind [in] 待设置数据
 * @return SnmpBerResult 操作结果
 */
SnmpBerResult agent_mib_set(const SnmpOid* oid, const SnmpVarBind* varbind);

/**
 * @brief 处理SNMP GETNEXT请求（获取下一个有效节点）
 * @param oid [in] 当前查询OID
 * @param next_oid [out] 下一个节点OID
 * @param varbind [out] 下一个节点数据
 * @return SnmpBerResult 操作结果
 */
SnmpBerResult agent_mib_get_next(const SnmpOid* oid, SnmpOid* next_oid, SnmpVarBind* varbind);

/**
 * @brief 处理SNMP GETBULK请求（批量获取节点）
 * @param start_oid [in] 起始OID
 * @param max_repetitions [in] 最大获取次数
 * @param varbinds [out] 批量数据数组
 * @param out_count [out] 实际获取数量
 * @return SnmpBerResult 操作结果
 */
SnmpBerResult agent_mib_get_bulk(const SnmpOid* start_oid, uint8 max_repetitions,
                                SnmpVarBind* varbinds, uint8* out_count);

/*****************************************
 * 扩展函数声明（数据更新与回调）
 *****************************************/
/**
 * @brief 手动更新单个MIB节点数据
 * @param oid [in] 目标节点OID
 * @param value [in] 待更新数据
 */
void agent_mib_update(const SnmpOid* oid, const SnmpVarBind* value);

/**
 * @brief 批量更新所有MIB节点数据（强制全量刷新）
 */
void agent_mib_update_all(void);
uint32 agent_mib_get_total_nodes(void);
SnmpBerResult agent_mib_get_first(SnmpOid* oid, SnmpVarBind* value);

#endif // AGENT_MIB_H
    
