﻿#ifndef MIB_REGISTRY_H
#define MIB_REGISTRY_H

#include "agent_mib.h"  // 引入统一的MibNodeConfig定义
#include "type.h"       // 替换<stdbool.h>为自定义类型头文件

// 最大MIB节点数量（与agent_mib.h保持一致）
#ifndef MAX_MIB_NODES
#define MAX_MIB_NODES 32
#endif

/*****************************************
 * MIB节点注册管理函数（负责节点存储与查询）
 *****************************************/

/**
 * @brief 初始化MIB注册中心
 * @note 系统启动时调用，初始化节点存储数组
 */
void mib_registry_init(void);

/**
 * @brief 注册MIB节点（客户调用此函数添加自定义节点）
 * @param config [in] 节点配置（需包含OID、类型、权限及回调）
 * @return bool8 注册成功返回BOOL8_TRUE，节点满或OID重复返回BOOL8_FALSE
 */
bool8 mib_registry_add(const MibNodeConfig* config);

/**
 * @brief 根据OID查找MIB节点
 * @param oid [in] 待查询的节点OID
 * @return const MibNodeConfig* 找到返回节点指针，未找到返回NULL
 */
const MibNodeConfig* mib_registry_find(const SnmpOid* oid);

/**
 * @brief 获取所有注册的MIB节点
 * @param out_count [out] 输出实际注册的节点数量
 * @return const MibNodeConfig* 返回节点数组首地址
 */
const MibNodeConfig* mib_registry_get_all(uint8* out_count);

/**
 * @brief 查找下一个有效MIB节点（用于GETNEXT操作）
 * @param current_oid [in] 当前OID
 * @return const MibNodeConfig* 找到返回下一个节点，无则返回NULL
 */
const MibNodeConfig* mib_registry_find_next(const SnmpOid* current_oid);

/**
 * @brief 直接获取snmpEngineID的值（用于SNMPv3 Discovery等特殊场景）
 * @details 此函数绕过通用的MIB处理流程，直接查找并调用snmpEngineID节点的读回调。
 * @param out_vb [out] 用于接收snmpEngineID值的变量绑定结构体指针。
 * @return bool8 成功返回BOOL8_TRUE，失败返回BOOL8_FALSE。
 */
bool8 mib_registry_get_snmp_engine_id(SnmpVarBind* out_vb);

void mib_registry_init_default_nodes(void);
uint8 mib_registry_get_count(void);
int32 oid_compare(const SnmpOid* oid1, const SnmpOid* oid2);
void oid_to_string(const SnmpOid* oid, char* buf, uint16 buf_len);


#endif // MIB_REGISTRY_H
