﻿/*********************************************************************
 * @file    platform.h
 * @brief   平台接口头文件（包含中断保护函数声明）
 *********************************************************************/
#ifndef PLATFORM_H
#define PLATFORM_H

#include "type.h"

// -------------------------- 类型定义 --------------------------
// 中断回调函数类型
typedef void (*PlatformIrqCallback)(void);

// -------------------------- 宏定义 --------------------------
// 定时器分频系数（0.1ms和1ms回调触发条件）
#ifndef TICK_0_1MS_DIV  // 修正：宏名称更清晰
#define TICK_0_1MS_DIV 100u  // 100次1μs中断 = 100μs = 0.1ms (修正：值从10改为100)
#endif

#ifndef TICK_1MS_DIV
#define TICK_1MS_DIV 1000u  // 1000次1μs中断 = 1ms
#endif

// -------------------------- 中断保护接口（核心！） --------------------------
/**
 * @brief 保存中断状态并关闭全局中断（临界区入口）
 * @return 原中断状态（用于后续恢复）
 */
extern uint32 platform_irq_save(void);

/**
 * @brief 恢复中断状态（临界区出口）
 * @param primask 由platform_irq_save返回的原中断状态
 */
extern void platform_irq_restore(uint32 primask);

// -------------------------- 随机数接口（加密必需！） --------------------------
/**
 * @brief 生成安全随机数（符合加密级要求）
 * @param buf 输出缓冲区
 * @param len 需生成的随机数长度（字节）
 * @return 成功返回BOOL8_TRUE，失败返回BOOL8_FALSE
 */
//extern bool8 platform_generate_random(uint8* buf, uint32 len);

// -------------------------- 内存对齐接口（适配硬件要求） --------------------------
/**
 * @brief 分配指定对齐方式的内存（用于加密算法硬件加速）
 * @param size 内存大小（字节）
 * @param align 对齐要求（必须是2的幂，如4/8/16）
 * @return 成功返回对齐后的内存指针，失败返回NULL
 */
//extern void* platform_malloc_aligned(uint32 size, uint32 align);

/**
 * @brief 释放对齐内存（必须与platform_malloc_aligned配对使用）
 * @param ptr 由platform_malloc_aligned返回的内存指针
 */
//extern void platform_free_aligned(void* ptr);

/**
 * @brief 简化接口：分配8字节对齐内存（加密模块常用）
 * @param size 内存大小（字节）
 * @return 成功返回8字节对齐指针，失败返回NULL
 */
//extern void* platform_malloc_8aligned(uint32 size);

// -------------------------- 新增：字节序（大小端）转换接口 --------------------------
/**
 * @brief 32位主机序转大端序（网络字节序）
 * @param out 4字节输出缓冲区（存储大端结果）
 * @param value 待转换的32位主机序数值
 */
extern void platform_cpu_to_be32(uint8* out, uint32 value);

/**
 * @brief 32位大端序转主机序
 * @param in 4字节输入缓冲区（大端数据）
 * @return 转换后的32位主机序数值
 */
extern uint32 platform_be32_to_cpu(const uint8* in);

/**
 * @brief 16位主机序转大端序（网络字节序）
 * @param out 2字节输出缓冲区（存储大端结果）
 * @param value 待转换的16位主机序数值
 */
extern void platform_cpu_to_be16(uint8* out, uint16 value);

/**
 * @brief 16位大端序转主机序
 * @param in 2字节输入缓冲区（大端数据）
 * @return 转换后的16位主机序数值
 */
extern uint16 platform_be16_to_cpu(const uint8* in);
// -------------------------- 定时器/延时接口 --------------------------
/**
 * @brief 获取1μs精度时间戳
 * @return 64位时间戳（单位：μs）
 */
extern uint64 platform_get_tick_us(void); // 修正：函数名更清晰
extern uint32 platform_get_tick(void);

/**
 * @brief 获取毫秒精度时间戳
 * @return 32位时间戳（单位：ms）
 */
extern uint32 platform_get_tick_ms(void); // 新增：更清晰的函数名

/**
 * @brief 注册0.1ms周期回调函数
 * @param cb 回调函数指针（NULL表示取消）
 */
//extern void platform_register_0_1ms_callback(PlatformIrqCallback cb); // 修正：函数名更清晰

/**
 * @brief 注册1ms周期回调函数
 * @param cb 回调函数指针（NULL表示取消）
 */
extern void platform_register_1ms_callback(PlatformIrqCallback cb);

/**
 * @brief 毫秒级延时
 * @param ms 延时时间（单位：ms）
 */
extern void platform_delay_ms(uint32 ms);

// -------------------------- 平台初始化与硬件信息 --------------------------
/**
 * @brief 平台初始化（统一入口，必须在所有接口前调用）
 */
extern void platform_init(void);

/**
 * @brief 获取设备MAC地址（测试用固定MAC，可替换为硬件读取）
 * @param mac_addr 6字节MAC地址输出缓冲区
 * @return 成功返回BOOL8_TRUE，失败返回BOOL8_FALSE
 */
//extern bool8 platform_get_mac(uint8* mac_addr);

//extern void* platform_malloc(uint32 size);
//extern void platform_free(void* ptr);

#endif  // PLATFORM_H
